package com.meidusa.fastjson.parser.deserializer;

import static com.meidusa.fastjson.parser.JSONToken.EOF;
import static com.meidusa.fastjson.parser.JSONToken.FALSE;
import static com.meidusa.fastjson.parser.JSONToken.LBRACE;
import static com.meidusa.fastjson.parser.JSONToken.LBRACKET;
import static com.meidusa.fastjson.parser.JSONToken.LITERAL_FLOAT;
import static com.meidusa.fastjson.parser.JSONToken.LITERAL_INT;
import static com.meidusa.fastjson.parser.JSONToken.LITERAL_STRING;
import static com.meidusa.fastjson.parser.JSONToken.NEW;
import static com.meidusa.fastjson.parser.JSONToken.NULL;
import static com.meidusa.fastjson.parser.JSONToken.TRUE;

import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.HashMap;

import org.apache.commons.lang.StringUtils;

import com.meidusa.fastjson.JSONException;
import com.meidusa.fastjson.parser.DefaultExtJSONParser;
import com.meidusa.fastjson.parser.JSONLexer;
import com.meidusa.fastjson.parser.JSONScanner;
import com.meidusa.fastjson.parser.JSONToken;
import com.meidusa.fastjson.parser.ParserConfig;
import com.meidusa.fastmark.feature.Feature;

public class TypeHandleBeanDeserializer extends ASMJavaBeanDeserializer {

	private String key;

	private HashMap<String, Class<?>> valueClassMapping = new HashMap<String, Class<?>>();

	private ObjectDeserializer defaultAsmJavaBeanDeserializer;

	private boolean isAbstract = false;

	public void addValueClass(String str, Class<?> clazz) {
		valueClassMapping.put(str, clazz);
	}

	public String getKey() {
		return key;
	}

	public void setKey(String key) {
		this.key = key;
	}

	public HashMap<String, Class<?>> getValueClassMapping() {
		return valueClassMapping;
	}

	public void setValueClassMapping(HashMap<String, Class<?>> valueClassMapping) {
		this.valueClassMapping = valueClassMapping;
	}

	public TypeHandleBeanDeserializer(ParserConfig mapping, Class<?> clazz) {
		super(mapping, clazz);
		if (Modifier.isAbstract(clazz.getModifiers()) || Modifier.isInterface(clazz.getModifiers())) {
			defaultAsmJavaBeanDeserializer = null;
			isAbstract = true;
		} else {
			try {
				this.defaultAsmJavaBeanDeserializer = ASMDeserializerFactory.getInstance().createJavaBeanDeserializer(
						mapping, clazz);
			} catch (Exception e) {
			}
		}
		mapping.putDeserializer(clazz, this);
	}

	@Override
	public Object createInstance(DefaultExtJSONParser parser, Type type) {
		// do not implemented
		return new Object();
	}

	@Override
	public <T> T deserialze(DefaultExtJSONParser parser, Type type) {

		JSONScanner jsonscanner = (JSONScanner) parser.getLexer();
		int i = jsonscanner.getBufferPosition();
		char c = jsonscanner.getCurrent();
		int j = jsonscanner.token();
		String keyStr = null;

		Class<?> shouldBe = null;

		if (jsonscanner.token() == JSONToken.NULL) {
			jsonscanner.nextToken(JSONToken.COMMA);
			return null;
		}

		for (;;) {

			String handleStr = jsonscanner.scanSymbol(parser.getSymbolTable());

			if (handleStr == null) {
				if (jsonscanner.token() == JSONToken.RBRACE) {
					jsonscanner.nextToken(JSONToken.COMMA);
					break;
				}
				if (jsonscanner.token() == JSONToken.COMMA) {
					if (parser.isEnabled(Feature.AllowArbitraryCommas)) {
						continue;
					}
				}
			}

			jsonscanner.nextTokenWithColon();
			Object o = this.skipComplicatedValue(parser, jsonscanner);
			if (o != null && handleStr.equals(this.key)) {
				keyStr = o.toString();
				shouldBe = this.valueClassMapping.get(keyStr);
			}
			if (shouldBe != null)
				break;
			if (jsonscanner.token() == JSONToken.COMMA) {
				continue;
			}

			if (jsonscanner.token() == JSONToken.RBRACE) {
				jsonscanner.nextToken(JSONToken.COMMA);
				break;
			}

			if (jsonscanner.token() == JSONToken.IDENTIFIER || jsonscanner.token() == JSONToken.ERROR) {
				throw new JSONException("syntax error, unexpect token " + JSONToken.name(jsonscanner.token()));
			}

		}
		jsonscanner.reset(i, c, j);
		if (shouldBe == null) {
			if (isAbstract) {
				if (StringUtils.isEmpty(keyStr)) {
					throw new JSONException(key + " must be set");
				} else {
					throw new JSONException(key + " is :\"" + keyStr + "\" but it's matching class not found.");

				}
			} else {
				if (defaultAsmJavaBeanDeserializer != null) {
					return defaultAsmJavaBeanDeserializer.deserialze(parser, type);
				} else {
					return super.deserialze(parser, type);
				}
			}
		} else {
			return (T) parser.parseObject(shouldBe);
		}
	}

	public Object skipComplicatedValue(DefaultExtJSONParser parser, JSONLexer lexer) {
		switch (lexer.token()) {
		case LBRACKET: {
			int inner = 1;
			lexer.nextToken();
			for (;;) {
				if (lexer.token() == JSONToken.LBRACKET) {
					inner++;
				} else if (lexer.token() == JSONToken.RBRACKET) {
					if (--inner == 0) {
						break;
					}
				}
				lexer.nextToken();
			}
			lexer.nextToken();
			return null;
		}
		case LBRACE: {
			int inner = 1;
			lexer.nextToken();
			for (;;) {
				if (lexer.token() == JSONToken.LBRACE) {
					inner++;
				} else if (lexer.token() == JSONToken.RBRACE) {
					if (--inner == 0) {
						break;
					}
				}
				lexer.nextToken();
			}
			lexer.nextToken();
			return null;
		}
		case LITERAL_INT:
			Number intValue = lexer.integerValue();
			lexer.nextToken();
			return intValue;
		case LITERAL_FLOAT:
			Object value;
			if (parser.isEnabled(Feature.UseBigDecimal)) {
				value = lexer.decimalValue();
			} else {
				value = lexer.doubleValue();
			}
			lexer.nextToken();
			return value;
		case LITERAL_STRING:
			String stringLiteral = lexer.stringVal();
			lexer.nextToken(JSONToken.COMMA);

			if (lexer.isEnabled(Feature.AllowISO8601DateFormat)) {
				JSONScanner iso8601Lexer = new JSONScanner(stringLiteral);
				if (iso8601Lexer.scanISO8601DateIfMatch()) {
					return iso8601Lexer.getCalendar().getTime();
				}
			}

			return stringLiteral;
		case NULL:
			lexer.nextToken();
			return null;
		case TRUE:
			lexer.nextToken();
			return Boolean.TRUE;
		case FALSE:
			lexer.nextToken();
			return Boolean.FALSE;
		case NEW:
			lexer.nextToken(JSONToken.COMMA);
		case EOF:
			if (lexer.isBlankInput()) {
				return null;
			}
		default:
			throw new JSONException("TODO " + JSONToken.name(lexer.token()) + " " + lexer.stringVal());
		}
	}

}
