/*
 * Decompiled with CFR 0.152.
 */
package org.compass.core.lucene.engine.store.localcache;

import java.io.IOException;
import java.util.Set;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.index.IndexFileNameFilter;
import org.apache.lucene.index.LuceneFileNames;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.DirectoryWrapper;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.LockFactory;
import org.compass.core.CompassException;
import org.compass.core.config.CompassSettings;
import org.compass.core.lucene.engine.store.localcache.LocalCacheManager;
import org.compass.core.transaction.context.TransactionContextCallback;
import org.compass.core.util.StringUtils;
import org.compass.core.util.concurrent.ConcurrentHashSet;
import org.compass.core.util.concurrent.ConcurrentLinkedHashMap;

public class MemoryDirectoryCache
extends Directory
implements DirectoryWrapper {
    private static final Log logger = LogFactory.getLog(MemoryDirectoryCache.class);
    private final Directory dir;
    private final LocalCacheManager localCacheManager;
    private final Set<String> localFileNames = new ConcurrentHashSet<String>();
    private final boolean cacheFileNames;
    private final ConcurrentLinkedHashMap<CacheKey, byte[]> cache;
    private final int bucketSize;
    private ScheduledFuture cleanupTaskFuture;
    private final boolean isCompoundFile;

    public MemoryDirectoryCache(String connectionString, Directory dir, LocalCacheManager localCacheManager) {
        this.dir = dir;
        this.localCacheManager = localCacheManager;
        String[] args = StringUtils.delimitedListToStringArray(connectionString, "&");
        int bucketSize = 1024;
        long size = CompassSettings.parseStringAsBytes("64m");
        boolean cacheFileNames = true;
        for (String arg : args) {
            if (arg.startsWith("bucketSize=")) {
                bucketSize = (int)CompassSettings.parseStringAsBytes(arg.substring("bucketSize=".length()));
                continue;
            }
            if (arg.startsWith("size=")) {
                size = CompassSettings.parseStringAsBytes(arg.substring("size=".length()));
                continue;
            }
            if (!arg.startsWith("cacheFileNames=")) continue;
            cacheFileNames = Boolean.parseBoolean(arg.substring("cacheFileNames=".length()));
        }
        this.cacheFileNames = cacheFileNames;
        this.bucketSize = bucketSize;
        int numberOfCacheEntries = (int)(size / (long)bucketSize);
        this.cache = new ConcurrentLinkedHashMap(ConcurrentLinkedHashMap.EvictionPolicy.SECOND_CHANCE, numberOfCacheEntries, new ConcurrentLinkedHashMap.EvictionListener[0]);
        this.isCompoundFile = localCacheManager == null ? false : localCacheManager.getSearchEngineFactory().getLuceneIndexManager().getStore().isUseCompoundFile();
        if (cacheFileNames) {
            this.cleanupTaskFuture = localCacheManager.getSearchEngineFactory().getExecutorManager().scheduleWithFixedDelay(new CleanupTask(), 10L, 10L, TimeUnit.SECONDS);
        }
    }

    public int getBucketSize() {
        return this.bucketSize;
    }

    public Directory getWrappedDirectory() {
        return this.dir;
    }

    public void clearWrapper() throws IOException {
        this.cache.clear();
    }

    public String[] list() throws IOException {
        return this.dir.list();
    }

    public boolean fileExists(String name) throws IOException {
        if (!this.cacheFileNames) {
            return this.dir.fileExists(name);
        }
        if (this.shouldPerformOperationOnActualDirectory(name)) {
            return this.dir.fileExists(name);
        }
        if (this.localFileNames.contains(name)) {
            return true;
        }
        boolean fileExists = this.dir.fileExists(name);
        if (fileExists) {
            this.localFileNames.add(name);
        }
        return fileExists;
    }

    public long fileModified(String name) throws IOException {
        return this.dir.fileModified(name);
    }

    public void touchFile(String name) throws IOException {
        this.dir.touchFile(name);
    }

    public void deleteFile(String name) throws IOException {
        this.dir.deleteFile(name);
    }

    public void renameFile(String from, String to) throws IOException {
        this.dir.renameFile(from, to);
    }

    public long fileLength(String name) throws IOException {
        return this.dir.fileLength(name);
    }

    public void close() throws IOException {
        if (this.cacheFileNames) {
            this.cleanupTaskFuture.cancel(true);
        }
        this.cache.clear();
        this.localFileNames.clear();
    }

    public Lock makeLock(String name) {
        return this.dir.makeLock(name);
    }

    public void clearLock(String name) throws IOException {
        this.dir.clearLock(name);
    }

    public void setLockFactory(LockFactory lockFactory) {
        this.dir.setLockFactory(lockFactory);
    }

    public LockFactory getLockFactory() {
        return this.dir.getLockFactory();
    }

    public String getLockID() {
        return this.dir.getLockID();
    }

    public IndexOutput createOutput(String name) throws IOException {
        return new WrappedIndexOutput(name, this.dir.createOutput(name));
    }

    public IndexInput openInput(String name) throws IOException {
        if (this.shouldWrapInput(name)) {
            return new WrappedIndexInput(name);
        }
        return this.dir.openInput(name);
    }

    public IndexInput openInput(String name, int bufferSize) throws IOException {
        if (this.shouldWrapInput(name)) {
            return new WrappedIndexInput(name);
        }
        return this.dir.openInput(name, bufferSize);
    }

    private boolean shouldWrapInput(String name) {
        if (this.shouldPerformOperationOnActualDirectory(name)) {
            return false;
        }
        return !this.isCompoundFile || !IndexFileNameFilter.getFilter().isCFSFile(name);
    }

    private boolean shouldPerformOperationOnActualDirectory(String name) {
        return LuceneFileNames.isStaticFile(name);
    }

    private class CacheKey {
        private final String fileName;
        private final long position;

        private CacheKey(String fileName, long position) {
            this.fileName = fileName;
            this.position = position;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            CacheKey cacheKey = (CacheKey)o;
            if (this.position != cacheKey.position) {
                return false;
            }
            return this.fileName.equals(cacheKey.fileName);
        }

        public int hashCode() {
            int result = this.fileName.hashCode();
            result = 31 * result + (int)(this.position ^ this.position >>> 32);
            return result;
        }
    }

    private class WrappedIndexInput
    extends IndexInput {
        private final String fileName;
        private IndexInput indexInput;
        private long currentPos = 0L;

        private WrappedIndexInput(String fileName) throws IOException {
            this.fileName = fileName;
            this.indexInput = MemoryDirectoryCache.this.dir.openInput(fileName, 1);
        }

        public byte readByte() throws IOException {
            CacheKey cacheKey = this.cacheKey(this.fileName, this.currentPos);
            byte[] cached = (byte[])MemoryDirectoryCache.this.cache.get(cacheKey);
            if (cached == null) {
                cached = this.readBytesForCache();
                MemoryDirectoryCache.this.cache.put(cacheKey, cached);
            }
            return cached[(int)(this.currentPos++ % (long)MemoryDirectoryCache.this.bucketSize)];
        }

        public void readBytes(byte[] b, int offset, int len) throws IOException {
            int sizeToRead;
            int indexInCache = (int)(this.currentPos % (long)MemoryDirectoryCache.this.bucketSize);
            do {
                CacheKey cacheKey = this.cacheKey(this.fileName, this.currentPos);
                byte[] cached = (byte[])MemoryDirectoryCache.this.cache.get(cacheKey);
                if (cached == null) {
                    cached = this.readBytesForCache();
                    MemoryDirectoryCache.this.cache.put(cacheKey, cached);
                }
                if ((sizeToRead = MemoryDirectoryCache.this.bucketSize - indexInCache) > len) {
                    sizeToRead = len;
                }
                System.arraycopy(cached, indexInCache, b, offset, sizeToRead);
                offset += sizeToRead;
                this.currentPos += (long)sizeToRead;
                indexInCache = 0;
            } while ((len -= sizeToRead) != 0);
        }

        public void readBytes(byte[] b, int offset, int len, boolean useBuffer) throws IOException {
            if (!useBuffer) {
                this.indexInput.seek(this.currentPos);
                this.indexInput.readBytes(b, offset, len, useBuffer);
                this.currentPos += (long)len;
            } else {
                this.readBytes(b, offset, len);
            }
        }

        public void close() throws IOException {
            this.indexInput.close();
        }

        public long getFilePointer() {
            return this.currentPos;
        }

        public void seek(long pos) throws IOException {
            this.currentPos = pos;
            this.indexInput.seek(pos);
        }

        public long length() {
            return this.indexInput.length();
        }

        public Object clone() {
            WrappedIndexInput clone = (WrappedIndexInput)((Object)super.clone());
            clone.indexInput = (IndexInput)this.indexInput.clone();
            return clone;
        }

        private byte[] readBytesForCache() throws IOException {
            this.indexInput.seek(this.currentPos - this.currentPos % (long)MemoryDirectoryCache.this.bucketSize);
            int size = (int)(this.length() - this.currentPos);
            if (size > MemoryDirectoryCache.this.bucketSize) {
                size = MemoryDirectoryCache.this.bucketSize;
            }
            byte[] cached = new byte[size];
            this.indexInput.readBytes(cached, 0, size, false);
            return cached;
        }

        private CacheKey cacheKey(String fileName, long currentPosition) {
            return new CacheKey(fileName, currentPosition - currentPosition % (long)MemoryDirectoryCache.this.bucketSize);
        }
    }

    private class WrappedIndexOutput
    extends IndexOutput {
        private final String fileName;
        private final IndexOutput indexOutput;

        private WrappedIndexOutput(String fileName, IndexOutput indexOutput) {
            this.fileName = fileName;
            this.indexOutput = indexOutput;
        }

        public void writeByte(byte b) throws IOException {
            this.indexOutput.writeByte(b);
        }

        public void writeBytes(byte[] b, int offset, int length) throws IOException {
            this.indexOutput.writeBytes(b, offset, length);
        }

        public void flush() throws IOException {
            this.indexOutput.flush();
        }

        public void close() throws IOException {
            this.indexOutput.close();
            MemoryDirectoryCache.this.localFileNames.add(this.fileName);
        }

        public long getFilePointer() {
            return this.indexOutput.getFilePointer();
        }

        public void seek(long pos) throws IOException {
            this.indexOutput.seek(pos);
        }

        public long length() throws IOException {
            return this.indexOutput.length();
        }
    }

    private class CleanupTask
    implements Runnable {
        private CleanupTask() {
        }

        public void run() {
            String[] fileNames = MemoryDirectoryCache.this.localCacheManager.getSearchEngineFactory().getTransactionContext().execute(new TransactionContextCallback<String[]>(){

                @Override
                public String[] doInTransaction() throws CompassException {
                    try {
                        return MemoryDirectoryCache.this.dir.list();
                    }
                    catch (IOException e) {
                        logger.error((Object)"Failed to list file names", (Throwable)e);
                        return null;
                    }
                }
            });
            if (fileNames == null) {
                return;
            }
            MemoryDirectoryCache.this.localFileNames.clear();
            for (String fileName : fileNames) {
                MemoryDirectoryCache.this.localFileNames.add(fileName);
            }
        }
    }
}

