/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.swift.snative;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.httpclient.Header;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.swift.exceptions.SwiftConfigurationException;
import org.apache.hadoop.fs.swift.exceptions.SwiftException;
import org.apache.hadoop.fs.swift.exceptions.SwiftInvalidResponseException;
import org.apache.hadoop.fs.swift.exceptions.SwiftOperationFailedException;
import org.apache.hadoop.fs.swift.http.HttpBodyContent;
import org.apache.hadoop.fs.swift.http.SwiftRestClient;
import org.apache.hadoop.fs.swift.snative.SwiftFileStatus;
import org.apache.hadoop.fs.swift.snative.SwiftObjectFileStatus;
import org.apache.hadoop.fs.swift.util.DurationStats;
import org.apache.hadoop.fs.swift.util.JSONUtil;
import org.apache.hadoop.fs.swift.util.SwiftObjectPath;
import org.apache.hadoop.fs.swift.util.SwiftUtils;
import org.codehaus.jackson.map.type.CollectionType;

public class SwiftNativeFileSystemStore {
    private static final Pattern URI_PATTERN = Pattern.compile("\"\\S+?\"");
    private static final String PATTERN = "EEE, d MMM yyyy hh:mm:ss zzz";
    private static final Log LOG = LogFactory.getLog(SwiftNativeFileSystemStore.class);
    private URI uri;
    private SwiftRestClient swiftRestClient;

    public void initialize(URI fsURI, Configuration configuration) throws IOException {
        this.uri = fsURI;
        this.swiftRestClient = SwiftRestClient.getInstance(fsURI, configuration);
    }

    public String toString() {
        return "SwiftNativeFileSystemStore with " + this.swiftRestClient;
    }

    public long getBlocksize() {
        return 1024L * this.swiftRestClient.getBlocksizeKB();
    }

    public long getPartsizeKB() {
        return this.swiftRestClient.getPartSizeKB();
    }

    public int getBufferSizeKB() {
        return this.swiftRestClient.getBufferSizeKB();
    }

    public int getThrottleDelay() {
        return this.swiftRestClient.getThrottleDelay();
    }

    public void uploadFile(Path path, InputStream inputStream, long length) throws IOException {
        this.swiftRestClient.upload(this.toObjectPath(path), inputStream, length, new Header[0]);
    }

    public void uploadFilePart(Path path, int partNumber, InputStream inputStream, long length) throws IOException {
        String stringPath = path.toUri().toString();
        String partitionFilename = SwiftUtils.partitionFilenameFromNumber(partNumber);
        stringPath = stringPath.endsWith("/") ? stringPath.concat(partitionFilename) : stringPath.concat("/").concat(partitionFilename);
        this.swiftRestClient.upload(new SwiftObjectPath(this.toDirPath(path).getContainer(), stringPath), inputStream, length, new Header[0]);
    }

    public void createManifestForPartUpload(Path path) throws IOException {
        String pathString = this.toObjectPath(path).toString();
        if (!pathString.endsWith("/")) {
            pathString = pathString.concat("/");
        }
        if (pathString.startsWith("/")) {
            pathString = pathString.substring(1);
        }
        this.swiftRestClient.upload(this.toObjectPath(path), new ByteArrayInputStream(new byte[0]), 0L, new Header("X-Object-Manifest", pathString));
    }

    public SwiftFileStatus getObjectMetadata(Path path) throws IOException {
        return this.getObjectMetadata(path, true);
    }

    public Header[] getObjectHeaders(Path path, boolean newest) throws IOException, FileNotFoundException {
        SwiftObjectPath objectPath = this.toObjectPath(path);
        return this.stat(objectPath, newest);
    }

    public SwiftFileStatus getObjectMetadata(Path path, boolean newest) throws IOException, FileNotFoundException {
        SwiftObjectPath objectPath = this.toObjectPath(path);
        Header[] headers = this.stat(objectPath, newest);
        if (headers.length == 0) {
            throw new FileNotFoundException("Not Found " + path.toUri());
        }
        boolean isDir = false;
        long length = 0L;
        long lastModified = 0L;
        for (Header header : headers) {
            String headerName = header.getName();
            if (headerName.equals("X-Container-Object-Count") || headerName.equals("X-Container-Bytes-Used")) {
                length = 0L;
                isDir = true;
            }
            if ("Content-Length".equals(headerName)) {
                length = Long.parseLong(header.getValue());
            }
            if (!"Last-Modified".equals(headerName)) continue;
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat(PATTERN);
            try {
                lastModified = simpleDateFormat.parse(header.getValue()).getTime();
            }
            catch (ParseException e) {
                throw new SwiftException("Failed to parse " + header.toString(), e);
            }
        }
        if (lastModified == 0L) {
            lastModified = System.currentTimeMillis();
        }
        Path correctSwiftPath = this.getCorrectSwiftPath(path);
        return new SwiftFileStatus(length, isDir, 1, this.getBlocksize(), lastModified, correctSwiftPath);
    }

    private Header[] stat(SwiftObjectPath objectPath, boolean newest) throws IOException {
        Header[] headers = newest ? this.swiftRestClient.headRequest("getObjectMetadata-newest", objectPath, SwiftRestClient.NEWEST) : this.swiftRestClient.headRequest("getObjectMetadata", objectPath, new Header[0]);
        return headers;
    }

    public HttpBodyContent getObject(Path path) throws IOException {
        return this.swiftRestClient.getData(this.toObjectPath(path), SwiftRestClient.NEWEST);
    }

    public HttpBodyContent getObject(Path path, long byteRangeStart, long length) throws IOException {
        return this.swiftRestClient.getData(this.toObjectPath(path), byteRangeStart, length);
    }

    private List<FileStatus> listDirectory(SwiftObjectPath path, boolean listDeep, boolean newest) throws IOException {
        byte[] bytes;
        ArrayList<FileStatus> files = new ArrayList<FileStatus>();
        Path correctSwiftPath = this.getCorrectSwiftPath(path);
        try {
            bytes = this.swiftRestClient.listDeepObjectsInDirectory(path, listDeep, new Header[0]);
        }
        catch (FileNotFoundException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("File/Directory not found " + path));
            }
            if (SwiftUtils.isRootDir(path)) {
                return Collections.emptyList();
            }
            throw e;
        }
        catch (SwiftInvalidResponseException e) {
            if (e.getStatusCode() == 204) {
                if (SwiftUtils.isRootDir(path)) {
                    return Collections.emptyList();
                }
                SwiftFileStatus stat = this.getObjectMetadata(correctSwiftPath, newest);
                if (stat.isDirectory()) {
                    return Collections.emptyList();
                }
                files.add(stat);
                return files;
            }
            throw e;
        }
        CollectionType collectionType = JSONUtil.getJsonMapper().getTypeFactory().constructCollectionType(List.class, SwiftObjectFileStatus.class);
        List fileStatusList = (List)JSONUtil.toObject(new String(bytes), collectionType);
        if (fileStatusList.isEmpty()) {
            SwiftFileStatus objectMetadata = this.getObjectMetadata(correctSwiftPath, newest);
            if (objectMetadata.isFile()) {
                files.add(objectMetadata);
            }
            return files;
        }
        for (SwiftObjectFileStatus status : fileStatusList) {
            if (status.getName() == null) continue;
            files.add(new SwiftFileStatus(status.getBytes(), status.getBytes() == 0L, 1, this.getBlocksize(), status.getLast_modified().getTime(), this.getCorrectSwiftPath(new Path(status.getName()))));
        }
        return files;
    }

    public FileStatus[] listSubPaths(Path path, boolean recursive, boolean newest) throws IOException {
        List<FileStatus> fileStatuses = this.listDirectory(this.toDirPath(path), recursive, newest);
        return fileStatuses.toArray(new FileStatus[fileStatuses.size()]);
    }

    public void createDirectory(Path path) throws IOException {
        this.innerCreateDirectory(this.toDirPath(path));
    }

    private void innerCreateDirectory(SwiftObjectPath swiftObjectPath) throws IOException {
        this.swiftRestClient.putRequest(swiftObjectPath, new Header[0]);
    }

    private SwiftObjectPath toDirPath(Path path) throws SwiftConfigurationException {
        return SwiftObjectPath.fromPath(this.uri, path, false);
    }

    private SwiftObjectPath toObjectPath(Path path) throws SwiftConfigurationException {
        return SwiftObjectPath.fromPath(this.uri, path);
    }

    public List<URI> getObjectLocation(Path path) throws IOException {
        byte[] objectLocation = this.swiftRestClient.getObjectLocation(this.toObjectPath(path), new Header[0]);
        if (objectLocation == null || objectLocation.length == 0) {
            return new LinkedList<URI>();
        }
        return SwiftNativeFileSystemStore.extractUris(new String(objectLocation), path);
    }

    public boolean deleteObject(Path path) throws IOException {
        SwiftObjectPath swiftObjectPath = this.toObjectPath(path);
        if (!SwiftUtils.isRootDir(swiftObjectPath)) {
            return this.swiftRestClient.delete(swiftObjectPath, new Header[0]);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)"Not deleting root directory entry");
        }
        return true;
    }

    public boolean rmdir(Path path) throws IOException {
        return this.deleteObject(path);
    }

    public boolean objectExists(Path path) throws IOException {
        return this.objectExists(this.toObjectPath(path));
    }

    public boolean objectExists(SwiftObjectPath path) throws IOException {
        try {
            Header[] headers = this.swiftRestClient.headRequest("objectExists", path, SwiftRestClient.NEWEST);
            return headers.length != 0;
        }
        catch (FileNotFoundException e) {
            return false;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void rename(Path src, Path dst) throws FileNotFoundException, SwiftOperationFailedException, IOException {
        boolean srcIsFile;
        SwiftFileStatus dstMetadata;
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("mv " + src + " " + dst));
        }
        boolean renamingOnToSelf = src.equals((Object)dst);
        SwiftObjectPath srcObject = this.toObjectPath(src);
        SwiftObjectPath destObject = this.toObjectPath(dst);
        if (SwiftUtils.isRootDir(srcObject)) {
            throw new SwiftOperationFailedException("cannot rename root dir");
        }
        SwiftFileStatus srcMetadata = this.getObjectMetadata(src);
        try {
            dstMetadata = this.getObjectMetadata(dst);
        }
        catch (FileNotFoundException e) {
            LOG.debug((Object)"Destination does not exist");
            dstMetadata = null;
        }
        Path srcParent = src.getParent();
        Path dstParent = dst.getParent();
        if (dstParent != null && !dstParent.equals((Object)srcParent)) {
            try {
                this.getObjectMetadata(dstParent);
            }
            catch (FileNotFoundException e) {
                LOG.debug((Object)("destination parent directory " + dstParent + " doesn't exist"));
                throw e;
            }
        }
        boolean destExists = dstMetadata != null;
        boolean destIsDir = destExists && SwiftUtils.isDirectory(dstMetadata);
        List<FileStatus> childStats = this.listDirectory(srcObject, true, true);
        boolean bl = srcIsFile = !srcMetadata.isDir();
        if (srcIsFile) {
            SwiftObjectPath destPath;
            block23: {
                if (destExists) {
                    if (destIsDir) {
                        destPath = this.toObjectPath(new Path(dst, src.getName()));
                        break block23;
                    } else {
                        if (!renamingOnToSelf) {
                            throw new SwiftOperationFailedException("cannot rename a file over one that already exists");
                        }
                        LOG.debug((Object)"Renaming file onto self: no-op => success");
                        return;
                    }
                }
                destPath = this.toObjectPath(dst);
            }
            int childCount = childStats.size();
            if (childCount == 0) {
                this.copyThenDeleteObject(srcObject, destPath);
                return;
            }
            SwiftUtils.debug(LOG, "Source file appears to be partitioned. copying file and deleting children", new Object[0]);
            this.copyObject(srcObject, destPath);
            Iterator<FileStatus> i$ = childStats.iterator();
            while (true) {
                if (!i$.hasNext()) {
                    this.swiftRestClient.delete(srcObject, new Header[0]);
                    return;
                }
                FileStatus stat = i$.next();
                SwiftUtils.debug(LOG, "Deleting partitioned file %s ", stat);
                this.deleteObject(stat.getPath());
            }
        }
        if (destExists && !destIsDir) {
            throw new SwiftOperationFailedException("the source is a directory, but not the destination");
        }
        Path targetPath = destExists ? new Path(dst, src.getName()) : dst;
        SwiftObjectPath targetObjectPath = this.toObjectPath(targetPath);
        if (srcObject.isEqualToOrParentOf(targetObjectPath)) {
            throw new SwiftOperationFailedException("cannot move a directory under itself");
        }
        LOG.info((Object)("mv  " + srcObject + " " + targetPath));
        this.logDirectory("Directory to copy ", srcObject, childStats);
        String srcURI = src.toUri().toString();
        int prefixStripCount = srcURI.length() + 1;
        for (FileStatus fileStatus : childStats) {
            Path copySourcePath = fileStatus.getPath();
            String copySourceURI = copySourcePath.toUri().toString();
            String copyDestSubPath = copySourceURI.substring(prefixStripCount);
            Path copyDestPath = new Path(targetPath, copyDestSubPath);
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("srcURI=" + srcURI + "; copySourceURI=" + copySourceURI + "; copyDestSubPath=" + copyDestSubPath + "; copyDestPath=" + copyDestPath));
            }
            SwiftObjectPath copyDestination = this.toObjectPath(copyDestPath);
            try {
                this.copyThenDeleteObject(this.toObjectPath(copySourcePath), copyDestination);
            }
            catch (FileNotFoundException e) {
                LOG.info((Object)("Skipping rename of " + copySourcePath));
            }
            this.throttle();
        }
        if (SwiftUtils.isRootDir(srcObject)) return;
        try {
            this.copyThenDeleteObject(srcObject, targetObjectPath);
            return;
        }
        catch (FileNotFoundException e) {
            LOG.warn((Object)"Source directory deleted during rename", (Throwable)e);
            this.innerCreateDirectory(destObject);
        }
    }

    private void logDirectory(String message, SwiftObjectPath objectPath, Iterable<FileStatus> statuses) {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(message + ": listing of " + objectPath));
            for (FileStatus fileStatus : statuses) {
                LOG.debug((Object)fileStatus.getPath());
            }
        }
    }

    public void copy(Path srcKey, Path dstKey) throws IOException {
        SwiftObjectPath srcObject = this.toObjectPath(srcKey);
        SwiftObjectPath destObject = this.toObjectPath(dstKey);
        this.swiftRestClient.copyObject(srcObject, destObject, new Header[0]);
    }

    private void copyThenDeleteObject(SwiftObjectPath srcObject, SwiftObjectPath destObject) throws IOException {
        this.copyObject(srcObject, destObject);
        this.swiftRestClient.delete(srcObject, new Header[0]);
    }

    private void copyObject(SwiftObjectPath srcObject, SwiftObjectPath destObject) throws IOException {
        if (srcObject.isEqualToOrParentOf(destObject)) {
            throw new SwiftException("Can't copy " + srcObject + " onto " + destObject);
        }
        boolean copySucceeded = this.swiftRestClient.copyObject(srcObject, destObject, new Header[0]);
        if (!copySucceeded) {
            throw new SwiftException("Copy of " + srcObject + " to " + destObject + "failed");
        }
    }

    public Path getCorrectSwiftPath(Path path) throws SwiftException {
        try {
            URI fullUri = new URI(this.uri.getScheme(), this.uri.getAuthority(), path.toUri().getPath(), null, null);
            return new Path(fullUri);
        }
        catch (URISyntaxException e) {
            throw new SwiftException("Specified path " + path + " is incorrect", e);
        }
    }

    private Path getCorrectSwiftPath(SwiftObjectPath path) throws SwiftException {
        try {
            URI fullUri = new URI(this.uri.getScheme(), this.uri.getAuthority(), path.getObject(), null, null);
            return new Path(fullUri);
        }
        catch (URISyntaxException e) {
            throw new SwiftException("Specified path " + path + " is incorrect", e);
        }
    }

    public static List<URI> extractUris(String json, Path path) throws SwiftOperationFailedException {
        Matcher matcher = URI_PATTERN.matcher(json);
        ArrayList<URI> result = new ArrayList<URI>();
        while (matcher.find()) {
            String s = matcher.group();
            String uri = s.substring(1, s.length() - 1);
            try {
                URI createdUri = URI.create(uri);
                result.add(createdUri);
            }
            catch (IllegalArgumentException e) {
                throw new SwiftOperationFailedException(String.format("could not convert \"%s\" into a URI. source: %s  first JSON: %s", uri, path, json.substring(0, 256)));
            }
        }
        return result;
    }

    public void throttle() throws InterruptedIOException {
        int throttleDelay = this.getThrottleDelay();
        if (throttleDelay > 0) {
            try {
                Thread.sleep(throttleDelay);
            }
            catch (InterruptedException e) {
                throw (InterruptedIOException)new InterruptedIOException(e.toString()).initCause(e);
            }
        }
    }

    public List<DurationStats> getOperationStatistics() {
        return this.swiftRestClient.getOperationStatistics();
    }

    public boolean delete(Path absolutePath, boolean recursive) throws IOException {
        Path swiftPath = this.getCorrectSwiftPath(absolutePath);
        SwiftUtils.debug(LOG, "Deleting path '%s' recursive=%b", absolutePath, recursive);
        boolean askForNewest = true;
        SwiftFileStatus fileStatus = this.getObjectMetadata(swiftPath, askForNewest);
        FileStatus[] statuses = this.listSubPaths(absolutePath, true, askForNewest);
        if (statuses == null) {
            SwiftUtils.debug(LOG, "Path '%s' has no status -it has 'gone away'", absolutePath, recursive);
            return false;
        }
        int filecount = statuses.length;
        SwiftUtils.debug(LOG, "Path '%s' %d status entries'", absolutePath, filecount);
        if (filecount == 0) {
            this.rmdir(absolutePath);
            return true;
        }
        if (LOG.isDebugEnabled()) {
            SwiftUtils.debug(LOG, SwiftUtils.fileStatsToString(statuses, "\n"), new Object[0]);
        }
        if (filecount == 1 && swiftPath.equals((Object)statuses[0].getPath())) {
            SwiftUtils.debug(LOG, "Deleting simple file %s", absolutePath);
            this.deleteObject(absolutePath);
            return true;
        }
        if (!fileStatus.isDir()) {
            LOG.debug((Object)"Multiple child entries but entry has data: assume partitioned");
        } else if (!recursive) {
            throw new SwiftOperationFailedException("Directory " + (Object)((Object)fileStatus) + " is not empty: " + SwiftUtils.fileStatsToString(statuses, "; "));
        }
        for (FileStatus entryStatus : statuses) {
            Path entryPath = entryStatus.getPath();
            try {
                boolean deleted = this.deleteObject(entryPath);
                if (!deleted) {
                    SwiftUtils.debug(LOG, "Failed to delete entry '%s'; continuing", entryPath);
                }
            }
            catch (FileNotFoundException e) {
                SwiftUtils.debug(LOG, "Path '%s' is no longer present; continuing", entryPath);
            }
            this.throttle();
        }
        SwiftUtils.debug(LOG, "Deleting base entry %s", absolutePath);
        this.deleteObject(absolutePath);
        return true;
    }
}

