/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.indices;

import jakarta.json.stream.JsonGenerator;
import java.util.Map;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.JsonpSerializable;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.opensearch._types.mapping.TypeMapping;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: indices._types.IndexState

@JsonpDeserializable
public class IndexState implements JsonpSerializable {
    private final Map<String, Alias> aliases;

    @Nullable
    private final TypeMapping mappings;

    @Nullable
    private final IndexSettings settings;

    @Nullable
    private final IndexSettings defaults;

    @Nullable
    private final String dataStream;

    // ---------------------------------------------------------------------------------------------

    private IndexState(Builder builder) {

        this.aliases = ApiTypeHelper.unmodifiable(builder.aliases);
        this.mappings = builder.mappings;
        this.settings = builder.settings;
        this.defaults = builder.defaults;
        this.dataStream = builder.dataStream;

    }

    public static IndexState of(Function<Builder, ObjectBuilder<IndexState>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * API name: {@code aliases}
     */
    public final Map<String, Alias> aliases() {
        return this.aliases;
    }

    /**
     * API name: {@code mappings}
     */
    @Nullable
    public final TypeMapping mappings() {
        return this.mappings;
    }

    /**
     * API name: {@code settings}
     */
    @Nullable
    public final IndexSettings settings() {
        return this.settings;
    }

    /**
     * Default settings, included when the request's <code>include_default</code> is
     * <code>true</code>.
     * <p>
     * API name: {@code defaults}
     */
    @Nullable
    public final IndexSettings defaults() {
        return this.defaults;
    }

    /**
     * API name: {@code data_stream}
     */
    @Nullable
    public final String dataStream() {
        return this.dataStream;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        if (ApiTypeHelper.isDefined(this.aliases)) {
            generator.writeKey("aliases");
            generator.writeStartObject();
            for (Map.Entry<String, Alias> item0 : this.aliases.entrySet()) {
                generator.writeKey(item0.getKey());
                item0.getValue().serialize(generator, mapper);

            }
            generator.writeEnd();

        }
        if (this.mappings != null) {
            generator.writeKey("mappings");
            this.mappings.serialize(generator, mapper);

        }
        if (this.settings != null) {
            generator.writeKey("settings");
            this.settings.serialize(generator, mapper);

        }
        if (this.defaults != null) {
            generator.writeKey("defaults");
            this.defaults.serialize(generator, mapper);

        }
        if (this.dataStream != null) {
            generator.writeKey("data_stream");
            generator.write(this.dataStream);

        }

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link IndexState}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<IndexState> {
        @Nullable
        private Map<String, Alias> aliases;

        @Nullable
        private TypeMapping mappings;

        @Nullable
        private IndexSettings settings;

        @Nullable
        private IndexSettings defaults;

        @Nullable
        private String dataStream;

        /**
         * API name: {@code aliases}
         * <p>
         * Adds all entries of <code>map</code> to <code>aliases</code>.
         */
        public final Builder aliases(Map<String, Alias> map) {
            this.aliases = _mapPutAll(this.aliases, map);
            return this;
        }

        /**
         * API name: {@code aliases}
         * <p>
         * Adds an entry to <code>aliases</code>.
         */
        public final Builder aliases(String key, Alias value) {
            this.aliases = _mapPut(this.aliases, key, value);
            return this;
        }

        /**
         * API name: {@code aliases}
         * <p>
         * Adds an entry to <code>aliases</code> using a builder lambda.
         */
        public final Builder aliases(String key, Function<Alias.Builder, ObjectBuilder<Alias>> fn) {
            return aliases(key, fn.apply(new Alias.Builder()).build());
        }

        /**
         * API name: {@code mappings}
         */
        public final Builder mappings(@Nullable TypeMapping value) {
            this.mappings = value;
            return this;
        }

        /**
         * API name: {@code mappings}
         */
        public final Builder mappings(Function<TypeMapping.Builder, ObjectBuilder<TypeMapping>> fn) {
            return this.mappings(fn.apply(new TypeMapping.Builder()).build());
        }

        /**
         * API name: {@code settings}
         */
        public final Builder settings(@Nullable IndexSettings value) {
            this.settings = value;
            return this;
        }

        /**
         * API name: {@code settings}
         */
        public final Builder settings(Function<IndexSettings.Builder, ObjectBuilder<IndexSettings>> fn) {
            return this.settings(fn.apply(new IndexSettings.Builder()).build());
        }

        /**
         * Default settings, included when the request's <code>include_default</code> is
         * <code>true</code>.
         * <p>
         * API name: {@code defaults}
         */
        public final Builder defaults(@Nullable IndexSettings value) {
            this.defaults = value;
            return this;
        }

        /**
         * Default settings, included when the request's <code>include_default</code> is
         * <code>true</code>.
         * <p>
         * API name: {@code defaults}
         */
        public final Builder defaults(Function<IndexSettings.Builder, ObjectBuilder<IndexSettings>> fn) {
            return this.defaults(fn.apply(new IndexSettings.Builder()).build());
        }

        /**
         * API name: {@code data_stream}
         */
        public final Builder dataStream(@Nullable String value) {
            this.dataStream = value;
            return this;
        }

        /**
         * Builds a {@link IndexState}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public IndexState build() {
            _checkSingleUse();

            return new IndexState(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link IndexState}
     */
    public static final JsonpDeserializer<IndexState> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        IndexState::setupIndexStateDeserializer
    );

    protected static void setupIndexStateDeserializer(ObjectDeserializer<IndexState.Builder> op) {

        op.add(Builder::aliases, JsonpDeserializer.stringMapDeserializer(Alias._DESERIALIZER), "aliases");
        op.add(Builder::mappings, TypeMapping._DESERIALIZER, "mappings");
        op.add(Builder::settings, IndexSettings._DESERIALIZER, "settings");
        op.add(Builder::defaults, IndexSettings._DESERIALIZER, "defaults");
        op.add(Builder::dataStream, JsonpDeserializer.stringDeserializer(), "data_stream");

    }

}
