/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch._types.query_dsl;

import jakarta.json.stream.JsonGenerator;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.JsonpSerializable;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: _types.query_dsl.IntervalsWildcard

@JsonpDeserializable
public class IntervalsWildcard implements IntervalsQueryVariant, IntervalsVariant, JsonpSerializable {
    @Nullable
    private final String analyzer;

    private final String pattern;

    @Nullable
    private final String useField;

    // ---------------------------------------------------------------------------------------------

    private IntervalsWildcard(Builder builder) {

        this.analyzer = builder.analyzer;
        this.pattern = ApiTypeHelper.requireNonNull(builder.pattern, this, "pattern");
        this.useField = builder.useField;

    }

    public static IntervalsWildcard of(Function<Builder, ObjectBuilder<IntervalsWildcard>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * IntervalsQuery variant kind.
     */
    @Override
    public IntervalsQuery.Kind _intervalsQueryKind() {
        return IntervalsQuery.Kind.Wildcard;
    }

    /**
     * Intervals variant kind.
     */
    @Override
    public Intervals.Kind _intervalsKind() {
        return Intervals.Kind.Wildcard;
    }

    /**
     * API name: {@code analyzer}
     */
    @Nullable
    public final String analyzer() {
        return this.analyzer;
    }

    /**
     * Required - API name: {@code pattern}
     */
    public final String pattern() {
        return this.pattern;
    }

    /**
     * API name: {@code use_field}
     */
    @Nullable
    public final String useField() {
        return this.useField;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        if (this.analyzer != null) {
            generator.writeKey("analyzer");
            generator.write(this.analyzer);

        }
        generator.writeKey("pattern");
        generator.write(this.pattern);

        if (this.useField != null) {
            generator.writeKey("use_field");
            generator.write(this.useField);

        }

    }

    public Builder toBuilder() {
        return new Builder().analyzer(analyzer).pattern(pattern).useField(useField);
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link IntervalsWildcard}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<IntervalsWildcard> {
        @Nullable
        private String analyzer;

        private String pattern;

        @Nullable
        private String useField;

        /**
         * API name: {@code analyzer}
         */
        public final Builder analyzer(@Nullable String value) {
            this.analyzer = value;
            return this;
        }

        /**
         * Required - API name: {@code pattern}
         */
        public final Builder pattern(String value) {
            this.pattern = value;
            return this;
        }

        /**
         * API name: {@code use_field}
         */
        public final Builder useField(@Nullable String value) {
            this.useField = value;
            return this;
        }

        /**
         * Builds a {@link IntervalsWildcard}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public IntervalsWildcard build() {
            _checkSingleUse();

            return new IntervalsWildcard(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link IntervalsWildcard}
     */
    public static final JsonpDeserializer<IntervalsWildcard> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        IntervalsWildcard::setupIntervalsWildcardDeserializer
    );

    protected static void setupIntervalsWildcardDeserializer(ObjectDeserializer<IntervalsWildcard.Builder> op) {

        op.add(Builder::analyzer, JsonpDeserializer.stringDeserializer(), "analyzer");
        op.add(Builder::pattern, JsonpDeserializer.stringDeserializer(), "pattern");
        op.add(Builder::useField, JsonpDeserializer.stringDeserializer(), "use_field");

    }

}
