/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.watcher;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import org.elasticsearch.ElasticsearchTimeoutException;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.engine.VersionConflictEngineException;
import org.elasticsearch.xpack.support.clock.Clock;
import org.elasticsearch.xpack.watcher.WatcherState;
import org.elasticsearch.xpack.watcher.execution.ExecutionService;
import org.elasticsearch.xpack.watcher.support.Exceptions;
import org.elasticsearch.xpack.watcher.support.WatcherIndexTemplateRegistry;
import org.elasticsearch.xpack.watcher.trigger.TriggerService;
import org.elasticsearch.xpack.watcher.watch.Watch;
import org.elasticsearch.xpack.watcher.watch.WatchLockService;
import org.elasticsearch.xpack.watcher.watch.WatchStatus;
import org.elasticsearch.xpack.watcher.watch.WatchStore;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;

public class WatcherService
extends AbstractComponent {
    private final Clock clock;
    private final TriggerService triggerService;
    private final Watch.Parser watchParser;
    private final WatchStore watchStore;
    private final WatchLockService watchLockService;
    private final ExecutionService executionService;
    private final WatcherIndexTemplateRegistry watcherIndexTemplateRegistry;
    final AtomicReference<WatcherState> state = new AtomicReference<WatcherState>(WatcherState.STOPPED);

    @Inject
    public WatcherService(Settings settings, Clock clock, TriggerService triggerService, WatchStore watchStore, Watch.Parser watchParser, ExecutionService executionService, WatchLockService watchLockService, WatcherIndexTemplateRegistry watcherIndexTemplateRegistry) {
        super(settings);
        this.clock = clock;
        this.triggerService = triggerService;
        this.watchStore = watchStore;
        this.watchParser = watchParser;
        this.watchLockService = watchLockService;
        this.executionService = executionService;
        this.watcherIndexTemplateRegistry = watcherIndexTemplateRegistry;
    }

    public void start(ClusterState clusterState) throws Exception {
        if (this.state.compareAndSet(WatcherState.STOPPED, WatcherState.STARTING)) {
            try {
                this.logger.debug("starting watch service...");
                this.watcherIndexTemplateRegistry.addTemplatesIfMissing();
                this.watchLockService.start();
                this.watchStore.start(clusterState);
                this.executionService.start(clusterState);
                this.triggerService.start(this.watchStore.activeWatches());
                this.state.set(WatcherState.STARTED);
                this.logger.debug("watch service has started");
            }
            catch (Exception e) {
                this.state.set(WatcherState.STOPPED);
                throw e;
            }
        } else {
            this.logger.debug("not starting watcher, because its state is [{}] while [{}] is expected", this.state, (Object)WatcherState.STOPPED);
        }
    }

    public boolean validate(ClusterState state) {
        return this.watchStore.validate(state) && this.executionService.validate(state);
    }

    public void stop() {
        if (this.state.compareAndSet(WatcherState.STARTED, WatcherState.STOPPING)) {
            this.logger.debug("stopping watch service...");
            this.triggerService.stop();
            this.executionService.stop();
            try {
                this.watchLockService.stop();
            }
            catch (ElasticsearchTimeoutException te) {
                this.logger.warn("error stopping WatchLockService", (Throwable)te);
            }
            this.watchStore.stop();
            this.state.set(WatcherState.STOPPED);
            this.logger.debug("watch service has stopped");
        } else {
            this.logger.debug("not stopping watcher, because its state is [{}] while [{}] is expected", this.state, (Object)WatcherState.STARTED);
        }
    }

    public WatchStore.WatchDelete deleteWatch(String id) {
        this.ensureStarted();
        WatchStore.WatchDelete delete = this.watchStore.delete(id);
        if (delete.deleteResponse().getResult() == DocWriteResponse.Result.DELETED) {
            this.triggerService.remove(id);
        }
        return delete;
    }

    public IndexResponse putWatch(String id, BytesReference watchSource, boolean active) throws IOException {
        this.ensureStarted();
        DateTime now = this.clock.nowUTC();
        Watch watch = this.watchParser.parseWithSecrets(id, false, watchSource, now);
        watch.setState(active, now);
        WatchStore.WatchPut result = this.watchStore.put(watch);
        if (result.previous() == null) {
            if (result.current().status().state().isActive()) {
                this.triggerService.add(result.current());
            }
        } else if (result.current().status().state().isActive()) {
            if (!result.previous().status().state().isActive()) {
                this.triggerService.add(result.current());
            } else if (!result.previous().trigger().equals(result.current().trigger())) {
                this.triggerService.add(result.current());
            }
        } else {
            this.triggerService.remove(result.current().id());
        }
        return result.indexResponse();
    }

    public Watch getWatch(String name) {
        return this.watchStore.get(name);
    }

    public WatcherState state() {
        return this.state.get();
    }

    public WatchStatus ackWatch(String id, String[] actionIds) throws IOException {
        Watch watch;
        this.ensureStarted();
        if (actionIds == null || actionIds.length == 0) {
            actionIds = new String[]{"_all"};
        }
        if ((watch = this.watchStore.get(id)) == null) {
            throw Exceptions.illegalArgument("watch [{}] does not exist", id);
        }
        if (watch.ack(this.clock.now(DateTimeZone.UTC), actionIds)) {
            try {
                this.watchStore.updateStatus(watch);
            }
            catch (IOException ioe) {
                throw Exceptions.ioException("failed to update the watch [{}] on ack", ioe, watch.id());
            }
            catch (VersionConflictEngineException vcee) {
                throw Exceptions.illegalState("failed to update the watch [{}] on ack, perhaps it was force deleted", vcee, watch.id());
            }
        }
        return new WatchStatus(watch.status());
    }

    public WatchStatus activateWatch(String id) throws IOException {
        return this.setWatchState(id, true);
    }

    public WatchStatus deactivateWatch(String id) throws IOException {
        return this.setWatchState(id, false);
    }

    WatchStatus setWatchState(String id, boolean active) throws IOException {
        this.ensureStarted();
        Watch watch = this.watchStore.get(id);
        if (watch == null) {
            throw Exceptions.illegalArgument("watch [{}] does not exist", id);
        }
        if (watch.setState(active, this.clock.nowUTC())) {
            try {
                this.watchStore.updateStatus(watch);
                if (active) {
                    this.triggerService.add(watch);
                } else {
                    this.triggerService.remove(watch.id());
                }
            }
            catch (IOException ioe) {
                throw Exceptions.ioException("failed to update the watch [{}] on ack", ioe, watch.id());
            }
            catch (VersionConflictEngineException vcee) {
                throw Exceptions.illegalState("failed to update the watch [{}] on ack, perhaps it was force deleted", vcee, watch.id());
            }
        }
        return new WatchStatus(watch.status());
    }

    public long watchesCount() {
        return this.watchStore.watches().size();
    }

    private void ensureStarted() {
        if (this.state.get() != WatcherState.STARTED) {
            throw new IllegalStateException("not started");
        }
    }

    public Map<String, Object> usageStats() {
        Map<String, Object> innerMap = this.executionService.usageStats();
        innerMap.putAll(this.watchStore.usageStats());
        return innerMap;
    }

    public void watchIndexDeletedOrClosed() {
        this.watchStore.clearWatchesInMemory();
        this.executionService.clearExecutions();
    }
}

