/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc.esnative;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.LatchedActionListener;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.ClearScrollRequest;
import org.elasticsearch.action.search.ClearScrollResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.action.update.UpdateRequestBuilder;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.ElasticsearchClient;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.ValidationException;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.engine.DocumentMissingException;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.xpack.security.InternalClient;
import org.elasticsearch.xpack.security.SecurityTemplateService;
import org.elasticsearch.xpack.security.action.realm.ClearRealmCacheRequest;
import org.elasticsearch.xpack.security.action.realm.ClearRealmCacheResponse;
import org.elasticsearch.xpack.security.action.user.ChangePasswordRequest;
import org.elasticsearch.xpack.security.action.user.DeleteUserRequest;
import org.elasticsearch.xpack.security.action.user.PutUserRequest;
import org.elasticsearch.xpack.security.authc.esnative.ReservedRealm;
import org.elasticsearch.xpack.security.authc.esnative.UserAndPassword;
import org.elasticsearch.xpack.security.authc.support.Hasher;
import org.elasticsearch.xpack.security.authc.support.SecuredString;
import org.elasticsearch.xpack.security.client.SecurityClient;
import org.elasticsearch.xpack.security.user.User;

public class NativeUsersStore
extends AbstractComponent
implements ClusterStateListener {
    private static final String USER_DOC_TYPE = "user";
    private static final String RESERVED_USER_DOC_TYPE = "reserved-user";
    private final Hasher hasher = Hasher.BCRYPT;
    private final AtomicReference<State> state = new AtomicReference<State>(State.INITIALIZED);
    private final InternalClient client;
    private final boolean isTribeNode;
    private volatile boolean securityIndexExists = false;
    private volatile boolean canWrite = false;

    public NativeUsersStore(Settings settings, InternalClient client) {
        super(settings);
        this.client = client;
        this.isTribeNode = !settings.getGroups("tribe", true).isEmpty();
    }

    public User getUser(String username) {
        if (this.state() != State.STARTED) {
            this.logger.trace("attempted to get user [{}] before service was started", (Object)username);
            return null;
        }
        UserAndPassword uap = this.getUserAndPassword(username);
        return uap == null ? null : uap.user();
    }

    public void getUsers(String[] userNames, ActionListener<Collection<User>> listener) {
        if (this.state() != State.STARTED) {
            this.logger.trace("attempted to get users before service was started");
            listener.onFailure((Exception)new IllegalStateException("users cannot be retrieved as native user service has not been started"));
            return;
        }
        Consumer<Exception> handleException = t -> {
            if (t instanceof IndexNotFoundException) {
                this.logger.trace("could not retrieve users because security index does not exist");
                listener.onResponse(Collections.emptyList());
            } else {
                listener.onFailure(t);
            }
        };
        if (userNames.length == 1) {
            String username = userNames[0];
            this.getUserAndPassword(username, (ActionListener<UserAndPassword>)ActionListener.wrap(uap -> listener.onResponse(uap == null ? Collections.emptyList() : Collections.singletonList(uap.user())), handleException::accept));
        } else {
            try {
                Object query = userNames == null || userNames.length == 0 ? QueryBuilders.matchAllQuery() : QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.idsQuery((String[])new String[]{USER_DOC_TYPE}).addIds(userNames));
                SearchRequest request = (SearchRequest)this.client.prepareSearch(new String[]{".security"}).setScroll(TimeValue.timeValueSeconds((long)10L)).setTypes(new String[]{USER_DOC_TYPE}).setQuery((QueryBuilder)query).setSize(1000).setFetchSource(true).request();
                request.indicesOptions().ignoreUnavailable();
                InternalClient.fetchAllByEntity((Client)this.client, request, listener, hit -> {
                    UserAndPassword u = this.transformUser(hit.getId(), hit.getSource());
                    return u != null ? u.user() : null;
                });
            }
            catch (Exception e) {
                this.logger.error(() -> new ParameterizedMessage("unable to retrieve users {}", (Object)Arrays.toString(userNames)), (Throwable)e);
                listener.onFailure(e);
            }
        }
    }

    private UserAndPassword getUserAndPassword(final String username) {
        final AtomicReference<Object> userRef = new AtomicReference<Object>(null);
        CountDownLatch latch = new CountDownLatch(1);
        this.getUserAndPassword(username, (ActionListener<UserAndPassword>)new LatchedActionListener((ActionListener)new ActionListener<UserAndPassword>(){

            public void onResponse(UserAndPassword user) {
                userRef.set(user);
            }

            public void onFailure(Exception t) {
                if (t instanceof IndexNotFoundException) {
                    NativeUsersStore.this.logger.trace(() -> new ParameterizedMessage("failed to retrieve user [{}] since security index does not exist", (Object)username), (Throwable)t);
                } else {
                    NativeUsersStore.this.logger.error(() -> new ParameterizedMessage("failed to retrieve user [{}]", (Object)username), (Throwable)t);
                }
            }
        }, latch));
        try {
            latch.await(30L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            this.logger.error("timed out retrieving user [{}]", (Object)username);
            return null;
        }
        return userRef.get();
    }

    private void getUserAndPassword(final String user, final ActionListener<UserAndPassword> listener) {
        try {
            GetRequest request = (GetRequest)this.client.prepareGet(".security", USER_DOC_TYPE, user).request();
            this.client.get(request, (ActionListener)new ActionListener<GetResponse>(){

                public void onResponse(GetResponse response) {
                    listener.onResponse((Object)NativeUsersStore.this.transformUser(response.getId(), response.getSource()));
                }

                public void onFailure(Exception t) {
                    if (t instanceof IndexNotFoundException) {
                        NativeUsersStore.this.logger.trace(() -> new ParameterizedMessage("could not retrieve user [{}] because security index does not exist", (Object)user), (Throwable)t);
                    } else {
                        NativeUsersStore.this.logger.error(() -> new ParameterizedMessage("failed to retrieve user [{}]", (Object)user), (Throwable)t);
                    }
                    listener.onResponse(null);
                }
            });
        }
        catch (IndexNotFoundException infe) {
            this.logger.trace("could not retrieve user [{}] because security index does not exist", (Object)user);
            listener.onResponse(null);
        }
        catch (Exception e) {
            this.logger.error(() -> new ParameterizedMessage("unable to retrieve user [{}]", (Object)user), (Throwable)e);
            listener.onFailure(e);
        }
    }

    public void changePassword(final ChangePasswordRequest request, final ActionListener<Void> listener) {
        final String username = request.username();
        assert (!"_system".equals(username) && !"_xpack".equals(username)) : username + "is internal!";
        if (this.state() != State.STARTED) {
            listener.onFailure((Exception)new IllegalStateException("password cannot be changed as user service has not been started"));
            return;
        }
        if (this.isTribeNode) {
            listener.onFailure((Exception)new UnsupportedOperationException("users may not be created or modified using a tribe node"));
            return;
        }
        if (!this.canWrite) {
            listener.onFailure((Exception)new IllegalStateException("password cannot be changed as user service cannot write until template and mappings are up to date"));
            return;
        }
        final String docType = ReservedRealm.isReserved(username, this.settings) ? RESERVED_USER_DOC_TYPE : USER_DOC_TYPE;
        ((UpdateRequestBuilder)this.client.prepareUpdate(".security", docType, username).setDoc(User.Fields.PASSWORD.getPreferredName(), (Object)String.valueOf(request.passwordHash())).setRefreshPolicy(request.getRefreshPolicy())).execute((ActionListener)new ActionListener<UpdateResponse>(){

            public void onResponse(UpdateResponse updateResponse) {
                assert (updateResponse.getResult() == DocWriteResponse.Result.UPDATED);
                NativeUsersStore.this.clearRealmCache(request.username(), listener, null);
            }

            public void onFailure(Exception e) {
                if (NativeUsersStore.isIndexNotFoundOrDocumentMissing(e)) {
                    if (docType.equals(NativeUsersStore.RESERVED_USER_DOC_TYPE)) {
                        NativeUsersStore.this.createReservedUser(username, request.passwordHash(), request.getRefreshPolicy(), (ActionListener<Void>)listener);
                    } else {
                        NativeUsersStore.this.logger.debug(() -> new ParameterizedMessage("failed to change password for user [{}]", (Object)request.username()), (Throwable)e);
                        ValidationException validationException = new ValidationException();
                        validationException.addValidationError("user must exist in order to change password");
                        listener.onFailure((Exception)validationException);
                    }
                } else {
                    listener.onFailure(e);
                }
            }
        });
    }

    private void createReservedUser(final String username, char[] passwordHash, WriteRequest.RefreshPolicy refresh, final ActionListener<Void> listener) {
        ((IndexRequestBuilder)this.client.prepareIndex(".security", RESERVED_USER_DOC_TYPE, username).setSource(User.Fields.PASSWORD.getPreferredName(), (Object)String.valueOf(passwordHash), User.Fields.ENABLED.getPreferredName(), (Object)true).setRefreshPolicy(refresh)).execute((ActionListener)new ActionListener<IndexResponse>(){

            public void onResponse(IndexResponse indexResponse) {
                NativeUsersStore.this.clearRealmCache(username, listener, null);
            }

            public void onFailure(Exception e) {
                listener.onFailure(e);
            }
        });
    }

    public void putUser(PutUserRequest request, ActionListener<Boolean> listener) {
        if (this.state() != State.STARTED) {
            listener.onFailure((Exception)new IllegalStateException("user cannot be added as native user service has not been started"));
            return;
        }
        if (this.isTribeNode) {
            listener.onFailure((Exception)new UnsupportedOperationException("users may not be created or modified using a tribe node"));
            return;
        }
        if (!this.canWrite) {
            listener.onFailure((Exception)new IllegalStateException("user cannot be created or changed as the user service cannot write until template and mappings are up to date"));
            return;
        }
        try {
            if (request.passwordHash() == null) {
                this.updateUserWithoutPassword(request, listener);
            } else {
                this.indexUser(request, listener);
            }
        }
        catch (Exception e) {
            this.logger.error(() -> new ParameterizedMessage("unable to put user [{}]", (Object)request.username()), (Throwable)e);
            listener.onFailure(e);
        }
    }

    private void updateUserWithoutPassword(final PutUserRequest putUserRequest, final ActionListener<Boolean> listener) {
        assert (putUserRequest.passwordHash() == null);
        ((UpdateRequestBuilder)this.client.prepareUpdate(".security", USER_DOC_TYPE, putUserRequest.username()).setDoc(new Object[]{User.Fields.USERNAME.getPreferredName(), putUserRequest.username(), User.Fields.ROLES.getPreferredName(), putUserRequest.roles(), User.Fields.FULL_NAME.getPreferredName(), putUserRequest.fullName(), User.Fields.EMAIL.getPreferredName(), putUserRequest.email(), User.Fields.METADATA.getPreferredName(), putUserRequest.metadata(), User.Fields.ENABLED.getPreferredName(), putUserRequest.enabled()}).setRefreshPolicy(putUserRequest.getRefreshPolicy())).execute((ActionListener)new ActionListener<UpdateResponse>(){

            public void onResponse(UpdateResponse updateResponse) {
                assert (updateResponse.getResult() == DocWriteResponse.Result.UPDATED);
                NativeUsersStore.this.clearRealmCache(putUserRequest.username(), listener, false);
            }

            public void onFailure(Exception e) {
                Exception failure = e;
                if (NativeUsersStore.isIndexNotFoundOrDocumentMissing(e)) {
                    NativeUsersStore.this.logger.debug(() -> new ParameterizedMessage("failed to update user document with username [{}]", (Object)putUserRequest.username()), (Throwable)e);
                    ValidationException validationException = new ValidationException();
                    validationException.addValidationError("password must be specified unless you are updating an existing user");
                    failure = validationException;
                }
                listener.onFailure(failure);
            }
        });
    }

    private void indexUser(final PutUserRequest putUserRequest, final ActionListener<Boolean> listener) {
        assert (putUserRequest.passwordHash() != null);
        ((IndexRequestBuilder)this.client.prepareIndex(".security", USER_DOC_TYPE, putUserRequest.username()).setSource(new Object[]{User.Fields.USERNAME.getPreferredName(), putUserRequest.username(), User.Fields.PASSWORD.getPreferredName(), String.valueOf(putUserRequest.passwordHash()), User.Fields.ROLES.getPreferredName(), putUserRequest.roles(), User.Fields.FULL_NAME.getPreferredName(), putUserRequest.fullName(), User.Fields.EMAIL.getPreferredName(), putUserRequest.email(), User.Fields.METADATA.getPreferredName(), putUserRequest.metadata(), User.Fields.ENABLED.getPreferredName(), putUserRequest.enabled()}).setRefreshPolicy(putUserRequest.getRefreshPolicy())).execute((ActionListener)new ActionListener<IndexResponse>(){

            public void onResponse(IndexResponse updateResponse) {
                NativeUsersStore.this.clearRealmCache(putUserRequest.username(), listener, updateResponse.getResult() == DocWriteResponse.Result.CREATED);
            }

            public void onFailure(Exception e) {
                listener.onFailure(e);
            }
        });
    }

    public void setEnabled(String username, boolean enabled, WriteRequest.RefreshPolicy refreshPolicy, ActionListener<Void> listener) {
        if (this.state() != State.STARTED) {
            listener.onFailure((Exception)new IllegalStateException("enabled status cannot be changed as native user service has not been started"));
            return;
        }
        if (this.isTribeNode) {
            listener.onFailure((Exception)new UnsupportedOperationException("users may not be created or modified using a tribe node"));
            return;
        }
        if (!this.canWrite) {
            listener.onFailure((Exception)new IllegalStateException("enabled status cannot be changed as user service cannot write until template and mappings are up to date"));
            return;
        }
        if (ReservedRealm.isReserved(username, this.settings)) {
            this.setReservedUserEnabled(username, enabled, refreshPolicy, listener);
        } else {
            this.setRegularUserEnabled(username, enabled, refreshPolicy, listener);
        }
    }

    private void setRegularUserEnabled(final String username, final boolean enabled, WriteRequest.RefreshPolicy refreshPolicy, final ActionListener<Void> listener) {
        try {
            ((UpdateRequestBuilder)this.client.prepareUpdate(".security", USER_DOC_TYPE, username).setDoc(User.Fields.ENABLED.getPreferredName(), (Object)enabled).setRefreshPolicy(refreshPolicy)).execute((ActionListener)new ActionListener<UpdateResponse>(){

                public void onResponse(UpdateResponse updateResponse) {
                    assert (updateResponse.getResult() == DocWriteResponse.Result.UPDATED);
                    NativeUsersStore.this.clearRealmCache(username, listener, null);
                }

                public void onFailure(Exception e) {
                    Exception failure = e;
                    if (NativeUsersStore.isIndexNotFoundOrDocumentMissing(e)) {
                        NativeUsersStore.this.logger.debug(() -> new ParameterizedMessage("failed to {} user [{}]", (Object)(enabled ? "enable" : "disable"), (Object)username), (Throwable)e);
                        ValidationException validationException = new ValidationException();
                        validationException.addValidationError("only existing users can be " + (enabled ? "enabled" : "disabled"));
                        failure = validationException;
                    }
                    listener.onFailure(failure);
                }
            });
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    private void setReservedUserEnabled(final String username, boolean enabled, WriteRequest.RefreshPolicy refreshPolicy, final ActionListener<Void> listener) {
        try {
            ((UpdateRequestBuilder)this.client.prepareUpdate(".security", RESERVED_USER_DOC_TYPE, username).setDoc(User.Fields.ENABLED.getPreferredName(), (Object)enabled).setUpsert(new Object[]{User.Fields.PASSWORD.getPreferredName(), String.valueOf(ReservedRealm.DEFAULT_PASSWORD_HASH), User.Fields.ENABLED.getPreferredName(), enabled}).setRefreshPolicy(refreshPolicy)).execute((ActionListener)new ActionListener<UpdateResponse>(){

                public void onResponse(UpdateResponse updateResponse) {
                    assert (updateResponse.getResult() == DocWriteResponse.Result.UPDATED || updateResponse.getResult() == DocWriteResponse.Result.CREATED);
                    NativeUsersStore.this.clearRealmCache(username, listener, null);
                }

                public void onFailure(Exception e) {
                    listener.onFailure(e);
                }
            });
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    public void deleteUser(final DeleteUserRequest deleteUserRequest, final ActionListener<Boolean> listener) {
        if (this.state() != State.STARTED) {
            listener.onFailure((Exception)new IllegalStateException("user cannot be deleted as native user service has not been started"));
            return;
        }
        if (this.isTribeNode) {
            listener.onFailure((Exception)new UnsupportedOperationException("users may not be deleted using a tribe node"));
            return;
        }
        if (!this.canWrite) {
            listener.onFailure((Exception)new IllegalStateException("user cannot be deleted as user service cannot write until template and mappings are up to date"));
            return;
        }
        try {
            DeleteRequest request = (DeleteRequest)this.client.prepareDelete(".security", USER_DOC_TYPE, deleteUserRequest.username()).request();
            request.indicesOptions().ignoreUnavailable();
            request.setRefreshPolicy(deleteUserRequest.getRefreshPolicy());
            this.client.delete(request, (ActionListener)new ActionListener<DeleteResponse>(){

                public void onResponse(DeleteResponse deleteResponse) {
                    NativeUsersStore.this.clearRealmCache(deleteUserRequest.username(), listener, deleteResponse.getResult() == DocWriteResponse.Result.DELETED);
                }

                public void onFailure(Exception e) {
                    listener.onFailure(e);
                }
            });
        }
        catch (Exception e) {
            this.logger.error("unable to remove user", (Throwable)e);
            listener.onFailure(e);
        }
    }

    public boolean canStart(ClusterState clusterState, boolean master) {
        if (this.state() != State.INITIALIZED) {
            return false;
        }
        if (clusterState.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            this.logger.debug("native users store waiting until gateway has recovered from disk");
            return false;
        }
        if (this.isTribeNode) {
            return true;
        }
        if (SecurityTemplateService.securityIndexMappingAndTemplateUpToDate(clusterState, this.logger)) {
            this.canWrite = true;
        } else if (SecurityTemplateService.securityIndexMappingAndTemplateSufficientToRead(clusterState, this.logger)) {
            this.canWrite = false;
        } else {
            this.canWrite = false;
            return false;
        }
        IndexMetaData metaData = clusterState.metaData().index(".security");
        if (metaData == null) {
            this.logger.debug("security index [{}] does not exist, so service can start", (Object)".security");
            return true;
        }
        if (clusterState.routingTable().index(".security").allPrimaryShardsActive()) {
            this.logger.debug("security index [{}] all primary shards started, so service can start", (Object)".security");
            this.securityIndexExists = true;
            return true;
        }
        return false;
    }

    public void start() {
        try {
            if (this.state.compareAndSet(State.INITIALIZED, State.STARTING)) {
                this.state.set(State.STARTED);
            }
        }
        catch (Exception e) {
            this.logger.error("failed to start native user store", (Throwable)e);
            this.state.set(State.FAILED);
        }
    }

    public void stop() {
        if (this.state.compareAndSet(State.STARTED, State.STOPPING)) {
            this.state.set(State.STOPPED);
        }
    }

    User verifyPassword(String username, SecuredString password) {
        if (this.state() != State.STARTED) {
            this.logger.trace("attempted to verify user credentials for [{}] but service was not started", (Object)username);
            return null;
        }
        UserAndPassword user = this.getUserAndPassword(username);
        if (user == null || user.passwordHash() == null) {
            return null;
        }
        if (this.hasher.verify(password, user.passwordHash())) {
            return user.user();
        }
        return null;
    }

    public boolean started() {
        return this.state() == State.STARTED;
    }

    boolean securityIndexExists() {
        return this.securityIndexExists;
    }

    ReservedUserInfo getReservedUserInfo(final String username) throws Exception {
        assert (this.started());
        final AtomicReference userInfoRef = new AtomicReference();
        final AtomicReference<Exception> failure = new AtomicReference<Exception>();
        CountDownLatch latch = new CountDownLatch(1);
        this.client.prepareGet(".security", RESERVED_USER_DOC_TYPE, username).execute((ActionListener)new LatchedActionListener((ActionListener)new ActionListener<GetResponse>(){

            public void onResponse(GetResponse getResponse) {
                if (getResponse.isExists()) {
                    Map sourceMap = getResponse.getSourceAsMap();
                    String password = (String)sourceMap.get(User.Fields.PASSWORD.getPreferredName());
                    Boolean enabled = (Boolean)sourceMap.get(User.Fields.ENABLED.getPreferredName());
                    if (password == null || password.isEmpty()) {
                        failure.set(new IllegalStateException("password hash must not be empty!"));
                    } else if (enabled == null) {
                        failure.set(new IllegalStateException("enabled must not be null!"));
                    } else {
                        userInfoRef.set(new ReservedUserInfo(password.toCharArray(), enabled));
                    }
                }
            }

            public void onFailure(Exception e) {
                if (e instanceof IndexNotFoundException) {
                    NativeUsersStore.this.logger.trace(() -> new ParameterizedMessage("could not retrieve built in user [{}] info since security index does not exist", (Object)username), (Throwable)e);
                } else {
                    NativeUsersStore.this.logger.error(() -> new ParameterizedMessage("failed to retrieve built in user [{}] info", (Object)username), (Throwable)e);
                    failure.set(e);
                }
            }
        }, latch));
        try {
            boolean responseReceived = latch.await(30L, TimeUnit.SECONDS);
            if (!responseReceived) {
                failure.set(new TimeoutException("timed out trying to get built in user [" + username + "]"));
            }
        }
        catch (InterruptedException e) {
            failure.set(e);
        }
        Exception failureCause = (Exception)failure.get();
        if (failureCause != null) {
            throw failureCause;
        }
        return (ReservedUserInfo)userInfoRef.get();
    }

    void getAllReservedUserInfo(final ActionListener<Map<String, ReservedUserInfo>> listener) {
        assert (this.started());
        this.client.prepareSearch(new String[]{".security"}).setTypes(new String[]{RESERVED_USER_DOC_TYPE}).setQuery((QueryBuilder)QueryBuilders.matchAllQuery()).setFetchSource(true).execute((ActionListener)new ActionListener<SearchResponse>(){

            public void onResponse(SearchResponse searchResponse) {
                HashMap<String, ReservedUserInfo> userInfos = new HashMap<String, ReservedUserInfo>();
                assert (searchResponse.getHits().getTotalHits() <= 10L) : "there are more than 10 reserved users we need to change this to retrieve them all!";
                for (SearchHit searchHit : searchResponse.getHits().getHits()) {
                    Map sourceMap = searchHit.getSource();
                    String password = (String)sourceMap.get(User.Fields.PASSWORD.getPreferredName());
                    Boolean enabled = (Boolean)sourceMap.get(User.Fields.ENABLED.getPreferredName());
                    if (password == null || password.isEmpty()) {
                        listener.onFailure((Exception)new IllegalStateException("password hash must not be empty!"));
                        break;
                    }
                    if (enabled == null) {
                        listener.onFailure((Exception)new IllegalStateException("enabled must not be null!"));
                        break;
                    }
                    userInfos.put(searchHit.getId(), new ReservedUserInfo(password.toCharArray(), enabled));
                }
                listener.onResponse(userInfos);
            }

            public void onFailure(Exception e) {
                if (e instanceof IndexNotFoundException) {
                    NativeUsersStore.this.logger.trace("could not retrieve built in users since security index does not exist", (Throwable)e);
                    listener.onResponse(Collections.emptyMap());
                } else {
                    NativeUsersStore.this.logger.error("failed to retrieve built in users", (Throwable)e);
                    listener.onFailure(e);
                }
            }
        });
    }

    private void clearScrollResponse(final String scrollId) {
        ClearScrollRequest clearScrollRequest = (ClearScrollRequest)this.client.prepareClearScroll().addScrollId(scrollId).request();
        this.client.clearScroll(clearScrollRequest, (ActionListener)new ActionListener<ClearScrollResponse>(){

            public void onResponse(ClearScrollResponse response) {
            }

            public void onFailure(Exception t) {
                NativeUsersStore.this.logger.warn(() -> new ParameterizedMessage("failed to clear scroll [{}]", (Object)scrollId), (Throwable)t);
            }
        });
    }

    private <Response> void clearRealmCache(final String username, final ActionListener<Response> listener, final Response response) {
        SecurityClient securityClient = new SecurityClient((ElasticsearchClient)this.client);
        ClearRealmCacheRequest request = (ClearRealmCacheRequest)securityClient.prepareClearRealmCache().usernames(username).request();
        securityClient.clearRealmCache(request, new ActionListener<ClearRealmCacheResponse>(){

            public void onResponse(ClearRealmCacheResponse nodes) {
                listener.onResponse(response);
            }

            public void onFailure(Exception e) {
                NativeUsersStore.this.logger.error(() -> new ParameterizedMessage("unable to clear realm cache for user [{}]", (Object)username), (Throwable)e);
                ElasticsearchException exception = new ElasticsearchException("clearing the cache for [" + username + "] failed. please clear the realm cache manually", (Throwable)e, new Object[0]);
                listener.onFailure((Exception)((Object)exception));
            }
        });
    }

    public void clusterChanged(ClusterChangedEvent event) {
        this.securityIndexExists = event.state().metaData().indices().get((Object)".security") != null;
        this.canWrite = SecurityTemplateService.securityIndexMappingAndTemplateUpToDate(event.state(), this.logger);
    }

    public State state() {
        return this.state.get();
    }

    public void reset() {
        State state = this.state();
        if (state != State.STOPPED && state != State.FAILED) {
            throw new IllegalStateException("can only reset if stopped!!!");
        }
        this.securityIndexExists = false;
        this.canWrite = false;
        this.state.set(State.INITIALIZED);
    }

    @Nullable
    private UserAndPassword transformUser(String username, Map<String, Object> sourceMap) {
        if (sourceMap == null) {
            return null;
        }
        try {
            String password = (String)sourceMap.get(User.Fields.PASSWORD.getPreferredName());
            String[] roles = ((List)sourceMap.get(User.Fields.ROLES.getPreferredName())).toArray(Strings.EMPTY_ARRAY);
            String fullName = (String)sourceMap.get(User.Fields.FULL_NAME.getPreferredName());
            String email = (String)sourceMap.get(User.Fields.EMAIL.getPreferredName());
            Boolean enabled = (Boolean)sourceMap.get(User.Fields.ENABLED.getPreferredName());
            if (enabled == null) {
                enabled = Boolean.TRUE;
            }
            Map metadata = (Map)sourceMap.get(User.Fields.METADATA.getPreferredName());
            return new UserAndPassword(new User(username, roles, fullName, email, metadata, enabled), password.toCharArray());
        }
        catch (Exception e) {
            this.logger.error(() -> new ParameterizedMessage("error in the format of data for user [{}]", (Object)username), (Throwable)e);
            return null;
        }
    }

    private static boolean isIndexNotFoundOrDocumentMissing(Exception e) {
        Throwable cause;
        return e instanceof ElasticsearchException && ((cause = ExceptionsHelper.unwrapCause((Throwable)e)) instanceof IndexNotFoundException || cause instanceof DocumentMissingException);
    }

    static class ReservedUserInfo {
        final char[] passwordHash;
        final boolean enabled;

        ReservedUserInfo(char[] passwordHash, boolean enabled) {
            this.passwordHash = passwordHash;
            this.enabled = enabled;
        }
    }

    public static enum State {
        INITIALIZED,
        STARTING,
        STARTED,
        STOPPING,
        STOPPED,
        FAILED;

    }
}

