/**
 * <pre>
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU AFFERO GENERAL PUBLIC LICENSE as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU AFFERO GENERAL PUBLIC LICENSE for more details. 
 * 	You should have received a copy of the GNU AFFERO GENERAL PUBLIC LICENSE along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * </pre>
 */
package com.meidusa.toolkit.net;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.meidusa.toolkit.common.heartbeat.HeartbeatDelayed;
import com.meidusa.toolkit.common.heartbeat.HeartbeatManager;
import com.meidusa.toolkit.common.heartbeat.Status;
import com.meidusa.toolkit.common.poolable.HashFunction;



/**
 * <pre>
 * 该Pool 提供负载均衡、failover、HA策略
 * 采用Load Balance ObjectPool，则object 必须实现{@link PoolableObject}
 * 默认提供2种负载均衡方案：
 * <li>轮询：请求将轮询分配到每个pool，每个pool的请求比较平均</li>
 * <li>繁忙程度：将所有Pool的Active Num做一个排序，最小的Active Num将优先分配请求</li>
 * &#064;author &lt;a href=mailto:piratebase@sina.com&gt;Struct chen&lt;/a&gt;
 * </pre>
 */
public class MultipleLoadBalanceBackendConnectionPool implements BackendConnectionPool {
	private static Logger logger = LoggerFactory.getLogger(MultipleLoadBalanceBackendConnectionPool.class);
	static int HashMod = Integer.valueOf(System.getProperty("venus.HashFunction.mod", "1024"));
	
	public static class DefaultFunction implements HashFunction{
		  
	    public int hash(Object key) {  

	    	if(key == null){
	    		return 0;
	    	}else{
	    		return Math.abs(key.hashCode()  % HashMod);
	    	}
	    }
	}
	
	static HashFunction function = new DefaultFunction();
	
	static {
		String functionName = System.getProperty("venus.HashFunction.class",DefaultFunction.class.getName());
		try {
			Class clazz =  Class.forName(functionName);
			function = (HashFunction)clazz.newInstance();
		} catch (Exception e) {
			logger.error("hashFunction="+functionName+" not found,using default:"+DefaultFunction.class.getName(),e);
		}
	}
	
    public static final int LOADBALANCING_ROUNDROBIN  = 1;
    public static final int LOADBALANCING_WEIGHTBASED = 2;
    public static final int LOADBALANCING_HA          = 3;
    public static final int LOADBALANCING_HASH  	  = 4;
    private boolean         enable;
    private final String name;
    private boolean closed = false;
    
    protected static class ActiveNumComparator implements Comparator<BackendConnectionPool> {

        public int compare(BackendConnectionPool o1, BackendConnectionPool o2) {
            return o1.getActive() - o2.getActive();
        }
    }



    /**
     * 负载均衡策略
     */
    private int                               loadbalance;

    private volatile long                        currentCount  = 0L;
    private BackendConnectionPool[]                      objectPools;

    private BackendConnectionPool[]                      runtimeObjectPools;

    private int index = 0;
    private ActiveNumComparator               comparator    = new ActiveNumComparator();
	private boolean valid;

	private SortedMap<Integer, BackendConnectionPool> circle = new TreeMap<Integer, BackendConnectionPool>(); 

    public MultipleLoadBalanceBackendConnectionPool(String name,int loadbalance, BackendConnectionPool... objectPools){
    	this.loadbalance = loadbalance;
    	this.name = name;
    	setObjectPools(objectPools);
    }

    public void setLoadbalance(int loadbalance) {
        this.loadbalance = loadbalance;
    }

    public BackendConnectionPool[] getObjectPools(){
    	return this.objectPools;
    }
    public void setObjectPools(BackendConnectionPool[] objectPools) {
    	this.objectPools = objectPools;
        this.runtimeObjectPools = objectPools.clone();
        
        if(runtimeObjectPools.length>0){
	    	for(int i=0;i < runtimeObjectPools.length;i++){
	    		
	    		circle.put(((HashMod / runtimeObjectPools.length) * i), runtimeObjectPools[i]);
	    	}
        }
    }

	public BackendConnectionPool getConsistenthashPool(Object key) {
		if (circle.isEmpty()) {
			return null;
		}
		int hash = function.hash(key!=null?key.toString():"");
		if (!circle.containsKey(hash)) {
			SortedMap<Integer, BackendConnectionPool> tailMap = circle.tailMap(hash);
			hash = tailMap.isEmpty() ? circle.firstKey() : tailMap.firstKey();
		}
		return circle.get(hash);
	}
    
    public BackendConnection borrowObject() throws Exception {
    	BackendConnectionPool pool = null;
        
        while(true){
        	final BackendConnectionPool[] poolsTemp = runtimeObjectPools;
	        if (poolsTemp.length == 0) {
	            throw new Exception("poolName="+name+", no valid pools");
	        }
	
	        if (loadbalance == LOADBALANCING_ROUNDROBIN) {
	            long current = currentCount++;
	            pool = poolsTemp[(int) (current % poolsTemp.length)];
	        } else if (loadbalance == LOADBALANCING_WEIGHTBASED) {
	            if (poolsTemp.length > 1) {
	            	BackendConnectionPool[] objectPoolsCloned = poolsTemp.clone();
	                Arrays.sort(objectPoolsCloned, comparator);
	                pool = objectPoolsCloned[0];
	            } else if (poolsTemp.length == 1) {
	                pool = poolsTemp[0];
	            }
	        } else if (loadbalance == LOADBALANCING_HA) {
	            // HA,只要有效的pool
	        	if(index < poolsTemp.length){
	        		pool = poolsTemp[index];
	        	}else{
	        		pool = poolsTemp[0];
	        	}
	        } else {
	            throw new Exception("poolName="+name+" loadbalance parameter error,parameter loadbalance in [1,2,3]");
	        }
	        
	        if(!pool.isValid() || pool.isClosed()){
	        	validate();
	        	continue;
	        }else{
	        	break;
	        }
        }
        
        return pool.borrowObject();
        

    }

    public int getActive() {
        int active = 0;
        for (BackendConnectionPool pool : objectPools) {
            active += pool.getActive();
        }
        return active;
    }

    public void deActive(BackendConnection obj) {
    	BackendConnection poolableObject = (BackendConnection) obj;
    	BackendConnectionPool pool = poolableObject.getPool();
        pool.deActive(obj);
    }

    public void returnObject(BackendConnection obj) {
    	BackendConnection poolableObject = (BackendConnection) obj;
    	BackendConnectionPool pool = poolableObject.getPool();
    	if(pool != null){
    		pool.returnObject(obj);
    	}
    }

    public int getIndex() {
		return index;
	}

	public void setIndex(int index) {
		this.index = index;
	}

	public boolean isEnable() {
        return enable;
    }

    public void setEnable(boolean isEnabled) {
        this.enable = isEnabled;
    }

	public boolean isValid() {
		return this.valid;
	}

	public void setValid(boolean valid) {
		this.valid = valid;
	}

	public class MultipleHeartbeatDelayed extends HeartbeatDelayed {

		public MultipleHeartbeatDelayed(long nsTime, TimeUnit timeUnit) {
			super(nsTime, timeUnit);
		}

		public boolean isCycle(){
			return true;
		}
		
		public Status doCheck() {
			if(MultipleLoadBalanceBackendConnectionPool.this.validate()){
				return Status.VALID;
			}else{
				return Status.INVALID;
			}
		}

		@Override
		public String getName() {
			return MultipleLoadBalanceBackendConnectionPool.this.getName();
		}
	}
	
	MultipleHeartbeatDelayed delayed;
	
	public void init() {
		delayed = new MultipleHeartbeatDelayed(1, TimeUnit.SECONDS);
		HeartbeatManager.addHeartbeat(delayed);
		for(BackendConnectionPool pool :this.objectPools){
			BackendConnection conn = null;
			try {
				conn = pool.borrowObject();
				if(conn instanceof AuthingableBackendConnection ){
					if(!((AuthingableBackendConnection) conn).isAuthenticated()  || conn.isClosed()){
						pool.setValid(false);
					}else{
						pool.setValid(true);
					}
				}else{
					if(conn.isClosed()){
						pool.setValid(false);
					}else{
						pool.setValid(true);
					}
				}
			} catch (Exception e) {
				logger.warn(e.getMessage(), e);
			}finally{
				if(conn != null){
					try {
						pool.returnObject(conn);
					} catch (Exception e) {
						logger.warn(e.getMessage(), e);
					}
				}
			}
		}
		
		validate();
	}

	public void close(){
		closed = true;
		HeartbeatManager.removeHeartbeat(delayed);
	 }
	 
	public synchronized boolean validate() {
		List<BackendConnectionPool> poolList = new ArrayList<BackendConnectionPool>();
		SortedMap<Integer, BackendConnectionPool> hashPool = new TreeMap<Integer, BackendConnectionPool>(); 
		
		for(int i=0;i < objectPools.length;i++){
			if(objectPools[i].isValid() && !objectPools[i].isClosed()){
				poolList.add(objectPools[i]);
				hashPool.put(((HashMod / objectPools.length) * i), objectPools[i]);
			}
    	}
		
		BackendConnectionPool[] poolsTemp = runtimeObjectPools = poolList.toArray(new BackendConnectionPool[poolList.size()]);
		SortedMap<Integer, BackendConnectionPool> oldPool = this.circle;
		this.circle = hashPool;
		oldPool.clear();
        if (poolsTemp.length == 0) {
            this.setValid(false);
            return false;
        }else{
        	this.setValid(true);
        	return true;
        }
	}

	public String getName() {
		return name;
	}

	@Override
	public boolean isClosed() {
		return closed;
	}
	
}
