/**
 * <pre>
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU AFFERO GENERAL PUBLIC LICENSE as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU AFFERO GENERAL PUBLIC LICENSE for more details. 
 * 	You should have received a copy of the GNU AFFERO GENERAL PUBLIC LICENSE along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * </pre>
 */
package com.meidusa.toolkit.net;

import java.io.IOException;
import java.nio.channels.ClosedSelectorException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


import com.meidusa.toolkit.net.config.ExceptionCodeConstant;
import com.meidusa.toolkit.net.util.LoopingThread;

public class ConnectionConnector extends LoopingThread {
    protected static final Logger LOGGER = LoggerFactory.getLogger(ConnectionConnector.class);
    private static final ConnectIdGenerator ID_GENERATOR = new ConnectIdGenerator();

    private final String name;
    private final Selector selector;
    private final BlockingQueue<BackendConnection> connectQueue;
    protected ConnectionManager[] processors;
    private int nextProcessor;
    private long connectCount;
    private int executorSize = Runtime.getRuntime().availableProcessors();
    
    public ConnectionConnector(String name) throws IOException {
        super.setName(name);
        this.name = name;
        this.selector = Selector.open();
        this.connectQueue = new LinkedBlockingQueue<BackendConnection>();
    }

    public long getConnectCount() {
        return connectCount;
    }

    public void setProcessors(ConnectionManager[] processors) {
        this.processors = processors;
    }
    
    public int getExecutorSize() {
		return executorSize;
	}

	public void setExecutorSize(int executorSize) {
		this.executorSize = executorSize;
	}

    public void postConnect(BackendConnection c) {
        connectQueue.offer(c);
        selector.wakeup();
    }
    
    public void initProcessors() throws IOException {
    	if(processors == null){
    		processors = new ConnectionManager[Runtime.getRuntime().availableProcessors()];
    		for(int i=0;i<processors.length ;i++){
    			try {
					processors[i] = new ConnectionManager(this.getName()+"-Manager-"+i, getExecutorSize());
					processors[i].start();
				} catch (IOException e) {
					LOGGER.error("create connection Manager error", e);
					this.shutdown();
				}
    		}
    	}
	}
    
    protected void willStart() {
    	super.willStart();
    	try {
    		initProcessors();
    	} catch (IOException e) {
			LOGGER.error("create connection Manager error", e);
			this.shutdown();
		}
    }

    @Override
    public void iterate() throws IOException {
        ++connectCount;
        selector.select(1000L);
        connect(selector);
        Set<SelectionKey> keys = selector.selectedKeys();
        try {
            for (SelectionKey key : keys) {
                Object att = key.attachment();
                if (att != null && key.isValid() && key.isConnectable()) {
                    finishConnect(key, att);
                } else {
                    key.cancel();
                }
            }
        } finally {
            keys.clear();
        }
    }

    protected void handleIterateFailure(Throwable e) {
    	if(e instanceof ClosedSelectorException){
    		super.handleIterateFailure(e);
    	}else if(e instanceof InterruptedException){
    		super.handleIterateFailure(e);
    	}else{
    		LOGGER.error(name, e);
    	}
    }
    
    private void connect(Selector selector) {
        BackendConnection c = null;
        while ((c = connectQueue.poll()) != null) {
            try {
                c.connect(selector);
            } catch (Throwable e) {
                c.handleError(ExceptionCodeConstant.ERR_CONNECT_SOCKET, e);
            }
        }
    }

    private void finishConnect(SelectionKey key, Object att) {
        BackendConnection c = (BackendConnection) att;
        try {
            if (c.finishConnect()) {
                clearSelectionKey(key);
                c.setId(ID_GENERATOR.getId());
                ConnectionManager processor = null;
                if(c.getProcessor() != null){
                	processor = c.getProcessor();
                }else{
                	processor = nextProcessor();
                }
                c.setProcessor(processor);
                processor.postRegister(c);
            }
        } catch (Throwable e) {
            clearSelectionKey(key);
            c.handleError(ExceptionCodeConstant.ERR_FINISH_CONNECT, e);
        }
    }

    private void clearSelectionKey(SelectionKey key) {
        if (key.isValid()) {
            key.attach(null);
            key.cancel();
        }
    }

    public int getFrontends(){
    	int i = 0;
    	for(ConnectionManager manager: processors){
    		i += manager.getFrontends();
    	}
    	return i;
    }
    
    public int getBackends(){
    	int i = 0;
    	for(ConnectionManager manager: processors){
    		i += manager.getBackends();
    	}
    	return i;
    }
    
    private ConnectionManager nextProcessor() {
        if (++nextProcessor == processors.length) {
            nextProcessor = 0;
        }
        return processors[nextProcessor];
    }

    private static class ConnectIdGenerator {

        private static final long MAX_VALUE = Long.MAX_VALUE;

        private long connectId = 0L;
        private final Object lock = new Object();

        private long getId() {
            synchronized (lock) {
                if (connectId >= MAX_VALUE) {
                    connectId = 0L;
                }
                return ++connectId;
            }
        }
    }

}
