/*
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU General Public License for more details. 
 * 	You should have received a copy of the GNU General Public License along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package com.meidusa.toolkit.net.util;

/**
 * A queue implementation that is more efficient than a wrapper around
 * java.util.Vector. Allows adding and removing elements to/from the beginning,
 * without the unneccessary System.arraycopy overhead of java.util.Vector.
 */
public class MemoryQueue<T> implements Queue<T> {
	public MemoryQueue(int suggestedSize) {
		_size = _suggestedSize = suggestedSize;
		_items = newArray(_size);
	}
	
	public MemoryQueue(int suggestedSize,int maxSize) {
		_size = _suggestedSize = suggestedSize;
		_items = newArray(_size);
		this.maxSize = maxSize;
	}

	public MemoryQueue() {
		this(4);
	}

	/* (non-Javadoc)
	 * @see com.meidusa.toolkit.net.util.Queue#clear()
	 */
	public synchronized void clear() {
		_count = _start = _end = 0;
		_size = _suggestedSize;
		_items = newArray(_size);
	}

	public synchronized boolean hasElements() {
		return (_count != 0);
	}

	/* (non-Javadoc)
	 * @see com.meidusa.toolkit.net.util.Queue#size()
	 */
	public synchronized int size() {
		return _count;
	}

	/* (non-Javadoc)
	 * @see com.meidusa.toolkit.net.util.Queue#prepend(T)
	 */
	public synchronized void prepend(T item) {
		if (_count == _size) {
			makeMoreRoom();
		}

		if (_start == 0) {
			_start = _size - 1;
		} else {
			_start--;
		}

		_items[_start] = item;
		_count++;
	}

	/* (non-Javadoc)
	 * @see com.meidusa.toolkit.net.util.Queue#append(T)
	 */
	public synchronized void append(T item) {
		// only notify if the queue was previously empty
		append0(item, false);
	}

	/**
	 * Internal append method. If subclassing queue, be sure to call this method
	 * from inside a synchronized block.
	 */
	protected void append0(T item, boolean notify) {
		
		while(_count >= maxSize && maxSize>0){
			try {
				this.wait(10);
			} catch (InterruptedException e) {
			}
		}
		
		if (_count == _size) {
			makeMoreRoom();
		}
		
		
		
		_items[_end] = item;
		_end = (_end + 1) % _size;
		_count++;

		if (notify) {
			notify();
		}
	}

	/* (non-Javadoc)
	 * @see com.meidusa.toolkit.net.util.Queue#getNonBlocking()
	 */
	public synchronized T getNonBlocking() {
		if (_count == 0) {
			return null;
		}

		// pull the object off, and clear our reference to it
		T retval = _items[_start];
		_items[_start] = null;

		_start = (_start + 1) % _size;
		_count--;
		return retval;
	}



	protected void makeMoreRoom() {
		T[] items = newArray(_size * 2);
		System.arraycopy(_items, _start, items, 0, _size - _start);
		System.arraycopy(_items, 0, items, _size - _start, _end);
		_start = 0;
		_end = _size;
		_size *= 2;
		_items = items;
	}

	// shrink by half
	protected void shrink() {
		T[] items = newArray(_size / 2);

		if (_start > _end) {
			// the data wraps around
			System.arraycopy(_items, _start, items, 0, _size - _start);
			System.arraycopy(_items, 0, items, _size - _start, _end + 1);

		} else {
			// the data does not wrap around
			System.arraycopy(_items, _start, items, 0, _end - _start + 1);
		}

		_size = _size / 2;
		_start = 0;
		_end = _count;
		_items = items;
	}

	@SuppressWarnings("unchecked")
	private T[] newArray(int size) {
		return (T[]) new Object[size];
	}

	public String toString() {
		StringBuilder buf = new StringBuilder();

		buf.append("[count=").append(_count);
		buf.append(", size=").append(_size);
		buf.append(", start=").append(_start);
		buf.append(", end=").append(_end);
		buf.append(", elements={");

		for (int i = 0; i < _count; i++) {
			int pos = (i + _start) % _size;
			if (i > 0)
				buf.append(", ");
			buf.append(_items[pos]);
		}

		return buf.append("}]").toString();
	}
	

	public int getMaxSize() {
		return maxSize;
	}

	public void setMaxSize(int maxSize) {
		this.maxSize = maxSize;
	}


	protected final static int MIN_SHRINK_SIZE = 1024;
	
	protected int maxSize = -1;
	protected T[] _items;
	protected int _count = 0;
	protected int _start = 0, _end = 0;
	protected int _suggestedSize, _size = 0;
}
