/**
 * <pre>
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU General Public License for more details. 
 * 	You should have received a copy of the GNU General Public License along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * </pre>
 */
package com.meidusa.toolkit.net;

import java.io.IOException;
import java.net.SocketException;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.spi.SelectorProvider;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutorService;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.meidusa.toolkit.common.bean.util.Initialisable;
import com.meidusa.toolkit.common.bean.util.InitialisationException;
import com.meidusa.toolkit.net.util.ConMgrStats;
import com.meidusa.toolkit.net.util.LoopingThread;
import com.meidusa.toolkit.net.util.MemoryQueue;
import com.meidusa.toolkit.net.util.Queue;
import com.meidusa.toolkit.net.util.Reporter;
import com.meidusa.toolkit.net.util.Tuple;

/**
 * @author <a href=mailto:piratebase@sina.com>Struct chen</a>
 */
public class ConnectionManager extends LoopingThread implements Reporter, Initialisable {

    protected static Logger                          logger                          = Logger.getLogger(ConnectionManager.class);
    protected static final int                       SELECT_LOOP_TIME                = 1000;

    // codes for notifyObservers()
    protected static final int                       CONNECTION_ESTABLISHED          = 0;

    protected static final int                       CONNECTION_FAILED               = 1;
    protected static final int                       CONNECTION_CLOSED               = 2;
    protected static final int                       CONNECTION_AUTHENTICATE_SUCCESS = 3;
    protected static final int                       CONNECTION_AUTHENTICATE_FAILD   = 4;
    protected Selector                               _selector;

    private ExecutorService                                 executor;

	private List<NetEventHandler>                    _handlers                       = new ArrayList<NetEventHandler>();
    protected ArrayList<ConnectionObserver>          _observers                      = new ArrayList<ConnectionObserver>();

    /** Our current runtime stats. */
    protected ConMgrStats                            _stats;

    /** ѾʧЧϿĶ */
    protected Queue<Tuple<Connection, Exception>>    _deathq                         = new MemoryQueue<Tuple<Connection, Exception>>();

    protected Queue<Tuple<NetEventHandler, Integer>> _registerQueue                  = new MemoryQueue<Tuple<NetEventHandler, Integer>>();

    /** Counts consecutive runtime errors in select() */
    protected int                                    _runtimeExceptionCount;

    /** Connection idle check per 5 second */
    protected long                                     idleCheckTime                   = 5000;

    private long                                     lastIdleCheckTime               = 0;
    
    /** queue size which message received and waiting for response (default: -1 -- no limit)  */
    protected long 								 maxQueueSizePerClient					= -1;
    /**
     * ÿмʱ ()
     * @param idleCheckTime
     */
    public void setIdleCheckTime(long idleCheckTime) {
        this.idleCheckTime = idleCheckTime;
    }

    public long getIdleCheckTime() {
		return idleCheckTime;
	}

	public long getMaxQueueSizePerClient() {
		return maxQueueSizePerClient;
	}

	public void setMaxQueueSizePerClient(long maxQueueSizePerClient) {
		this.maxQueueSizePerClient = maxQueueSizePerClient;
	}

	public void appendReport(StringBuilder report, long now, long sinceLast, boolean reset, Level level) {
        report.append("* ").append(this.getName()).append("\n");
        report.append("- Registed Connection size: ").append(_selector.keys().size()).append("\n");
        report.append("- created Connection size: ").append(_stats.connects.get()).append("\n");
        report.append("- disconnect Connection size: ").append(_stats.disconnects.get()).append("\n");
        if (reset) {
            _stats = new ConMgrStats();
        }
    }

    public ConnectionManager() throws IOException{
        _selector = SelectorProvider.provider().openSelector();
        // create our stats record
        _stats = new ConMgrStats();
    }

    public ConnectionManager(String managerName) throws IOException{
        super(managerName);
        _selector = SelectorProvider.provider().openSelector();

        // create our stats record
        _stats = new ConMgrStats();
        this.setDaemon(true);
    }
    
    
    protected void handleIterateFailure(Throwable e) {
    	logger.error("handle iterate error",e);
    }

    /**
     * Performs the select loop. This is the body of the conmgr thread.
     */
    protected void iterate() {
        final long iterStamp = System.currentTimeMillis();

        // رѾϿConnection
        Tuple<Connection, Exception> deathTuple;
        while ((deathTuple = _deathq.getNonBlocking()) != null) {
        	try{
        		deathTuple.left.close(deathTuple.right);
        	}catch(Exception e){
        		logger.error("close connection error conn="+deathTuple.left,e);
        	}
        }

        if (idleCheckTime > 0 && iterStamp - lastIdleCheckTime >= idleCheckTime) {
            lastIdleCheckTime = iterStamp;
            // رտʱ
            for (NetEventHandler handler : _handlers) {
                if (handler.checkIdle(iterStamp)) {
                	
                    // this will queue the connection for closure on our next tick
                    if (handler instanceof Connection) {
                    	Connection conn = (Connection)handler;
                    	long idleMillis = iterStamp - conn._lastEvent;
                    	logger.info("Disconnecting non-communicative connection [conn=" + conn + ",id="+conn.getId()+", manager="+this.getName()+", idle=" + idleMillis + "ms].");
                        closeConnection((Connection) handler, null);
                    }
                }else{
                	if(handler.needPing(iterStamp)){
                		handler.ping(iterStamp);
                	}
                }
                
                if(maxQueueSizePerClient > 0){
                	if (handler instanceof Connection) {
                    	Connection conn = (Connection)handler;
	                	if(handler.getQueueSize() > maxQueueSizePerClient){
	                		logger.info("Disconnecting message flood connection [conn=" + conn + ",id="+conn.getId()+", queueSize="+ handler.getQueueSize());
	                        closeConnection((Connection) handler, null);
	                	}
                	}
                }
            }
        }

        // עӼhandler map
        Tuple<NetEventHandler, Integer> registerHandler = null;
        while ((registerHandler = _registerQueue.getNonBlocking()) != null) {
            if (registerHandler.left instanceof Connection) {
                Connection connection = (Connection) registerHandler.left;
                this.registerConnection(connection, registerHandler.right.intValue());
                if(connection.getChannel() != null && connection.getChannel().isOpen()){
                	_handlers.add(connection);
                }
            } else {
                _handlers.add(registerHandler.left);
            }
        }

        // ¼
        Set<SelectionKey> ready = null;
        try {
            // check for incoming network events
            int ecount = _selector.select(SELECT_LOOP_TIME);

            ready = _selector.selectedKeys();

            if (ecount == 0) {
                if (ready.size() == 0) {
                    return;
                } else {
                    logger.warn("select() returned no selected sockets, but there are " + ready.size() + " in the ready set.");
                }
            }

        } catch (IOException ioe) {
            logger.warn("Failure select()ing.", ioe);
            return;
        } catch (RuntimeException re) {
            // instead of looping indefinitely after things go pear-shaped, shut
            // us down in an
            // orderly fashion
            logger.warn("Failure select()ing.", re);
            if (_runtimeExceptionCount++ >= 20) {
                logger.warn("Too many errors, bailing.");
                shutdown();
            }
            return;
        }
        // clear the runtime error count
        _runtimeExceptionCount = 0;

        // ¼ȣ
        for (SelectionKey selkey : ready) {
        	
        	try{
	            NetEventHandler handler = null;
	            handler = (NetEventHandler) selkey.attachment();
	            if (handler == null) {
	                logger.warn("Received network event but have no registered handler " + "[selkey=" + selkey + "].");
	                selkey.cancel();
	                continue;
	            }
	
	            if (selkey.isWritable()) {
	                try {
	                    handler.doWrite();
	                    
	                } catch (Exception e) {
	                    logger.warn("Error processing network data: " + handler + ".", e);
	                    if (handler != null && handler instanceof Connection) {
	                        closeConnection((Connection) handler, e);
	                    }
	                } 
	            }
	            
	            if (selkey.isReadable() || selkey.isAcceptable()) {
	                handler.handleEvent(iterStamp);
	            } 
        	}catch(java.nio.channels.CancelledKeyException e){
        	}
            
        }

        ready.clear();
    }

    /**
     * 첽ʽرһӡ رյӷdeathQueue
     */
    void closeConnection(Connection conn, Exception exception) {
        if (!conn.isClosed()) {
            _deathq.append(new Tuple<Connection, Exception>(conn, exception));
        }
    }

    public void closeAll() {
        synchronized (_selector) {
            Set<SelectionKey> keys = _selector.keys();
            for (SelectionKey key : keys) {
                Object object = key.attachment();
                if (object instanceof Connection) {
                    Connection conn = (Connection) object;
                    closeConnection(conn, null);
                }
            }
        }
    }

    /**
     *  ConnectionObserverConnection ص¼
     */
    public void addConnectionObserver(ConnectionObserver observer) {
        synchronized (_observers) {
            _observers.add(observer);
        }
    }

    /**
     *  Observer бɾһObserver
     */
    public void removeConnectionObserver(ConnectionObserver observer) {
        synchronized (_observers) {
            _observers.remove(observer);
        }
    }

    protected void notifyObservers(int code, Connection conn, Object arg1) {
        synchronized (_observers) {
            for (ConnectionObserver obs : _observers) {
                switch (code) {
                    case CONNECTION_ESTABLISHED:
                        obs.connectionEstablished(conn);
                        break;
                    case CONNECTION_FAILED:
                        obs.connectionFailed(conn, (Exception) arg1);
                        break;
                    case CONNECTION_CLOSED:
                        obs.connectionClosed(conn);
                        break;
                    default:
                        throw new RuntimeException("Invalid code supplied to notifyObservers: " + code);
                }
            }
        }
    }

    /**
     * 첽עһNetEventHandler
     */
    public void postRegisterNetEventHandler(NetEventHandler handler, int key) {
        _registerQueue.append(new Tuple<NetEventHandler, Integer>(handler, key));
        /**
         * ConnectionManagerڵȴselecṭܹ߳ٵĴregisterQueueеĶ
         */
        _selector.wakeup();
    }

    protected void afterRegisterConnection(Connection connection){
    	notifyObservers(CONNECTION_ESTABLISHED,connection,null);
    }
    
    /**
     * ConnectionManager һSocketChannel
     */
    protected void registerConnection(Connection connection, int key) {
    	SelectableChannel channel = connection.getChannel();
        if (logger.isDebugEnabled()) {
            logger.debug("[" + this.getName() + "] registed Connection[" + connection.getId() + "] connected!");
        }
        SelectionKey selkey = null;
        try {
            if (!(channel instanceof SelectableChannel)) {
                try {
                    logger.warn("Provided with un-selectable socket as result of accept(), can't " + "cope [channel=" + channel + "].");
                } catch (Error err) {
                    logger.warn("Un-selectable channel also couldn't be printed.");
                }
                // stick a fork in the socket
                if (channel != null) {
                    channel.close();
                }
                return;
            }

            channel.configureBlocking(false);
            selkey = channel.register(_selector, key, connection);
            connection.setConnectionManager(this);
        	connection.setExecutor(this.executor);
            connection.setSelectionKey(selkey);
            configConnection(connection);
            _stats.connects.incrementAndGet();
            connection.init();
            afterRegisterConnection(connection);
            _selector.wakeup();
            return;
        } catch (IOException ioe) {
            logger.error("register connection error: " + ioe);
        }

        if (selkey != null) {
            selkey.attach(null);
            selkey.cancel();
        }

        // make sure we don't leak a socket if something went awry
        if (channel != null) {
            try {
                channel.close();
            } catch (IOException ioe) {
                logger.warn("Failed closing aborted connection: " + ioe);
            }
        }
    }

    protected void configConnection(Connection connection) throws SocketException{};

    /**
     *  Connection رԺ
     */
    protected void connectionClosed(Connection conn) {
        /**
         * ɾرյض
         */
        _handlers.remove(conn);
        _stats.disconnects.incrementAndGet();
        /**
         * ֪ͨObserverбѾر
         */
        notifyObservers(CONNECTION_CLOSED, conn, null);
    }

    /**
     *  Connection 쳣Ժ
     */
    protected void connectionFailed(Connection conn, Exception ioe) {
        _handlers.remove(conn);
        _stats.disconnects.incrementAndGet();

        /**
         * 쳣ʱ֪ͨObservers
         */
        notifyObservers(CONNECTION_FAILED, conn, ioe);
    }

    public void setExecutor(ExecutorService executor) {
        this.executor = executor;
    }
    
    public ExecutorService getExecutor() {
        return this.executor;
    }

    public void init() throws InitialisationException {
    }
    
    protected void didShutdown() {
        ExecutorService executor = (ExecutorService) this.getExecutor();
        if (executor != null) {
            executor.shutdownNow();
        }
        super.didShutdown();
    }
    
    /**
     * get Current registered connections size
     * @return
     */
    public int getSize(){
    	return _handlers.size();
    }
}
