package com.meidusa.venus.backend.network;

import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.Date;

import org.apache.log4j.Logger;

import com.meidusa.toolkit.net.AuthingableConnection;
import com.meidusa.toolkit.net.Connection;
import com.meidusa.toolkit.net.authenticate.server.AuthResponseData;
import com.meidusa.toolkit.net.io.FramedInputStream;
import com.meidusa.toolkit.net.io.FramingOutputStream;
import com.meidusa.toolkit.net.io.PacketInputStream;
import com.meidusa.toolkit.net.io.PacketOutputStream;
import com.meidusa.venus.exception.CodedException;
import com.meidusa.venus.exception.VenusExceptionCodeConstant;
import com.meidusa.venus.io.VenusPacketMetaData;
import com.meidusa.venus.io.packet.AbstractServicePacket;
import com.meidusa.venus.io.packet.AuthenPacket;
import com.meidusa.venus.io.packet.AuthenPacketFactory;
import com.meidusa.venus.io.packet.ErrorPacket;
import com.meidusa.venus.io.packet.HandshakePacket;
import com.meidusa.venus.io.packet.OKPacket;
import com.meidusa.venus.io.packet.PacketConstant;
import com.meidusa.venus.io.packet.PingPacket;
import com.meidusa.venus.io.utils.StringUtil;

/**
 * 
 * @author Struct
 *
 */
public class VenusClientConnection extends AuthingableConnection {
    private static Logger logger = Logger.getLogger(VenusClientConnection.class);
    public static OKPacket OK = new OKPacket();
    private long lastPingTime = System.currentTimeMillis();
    private long lastPongTime = System.currentTimeMillis();
    private long pingInterval = 10 * 1000;
    private short serializeType = PacketConstant.CONTENT_TYPE_BSON;
    private int clientId;
    private boolean proxy;
    private static VenusPacketMetaData PACKET_META_DATA = new VenusPacketMetaData();
    public VenusClientConnection(SocketChannel channel, long createStamp) {
        super(channel, createStamp);
    }

    public short getProtocol() {
		return getSerializeType();
	}

	public int getClientId() {
		return clientId;
	}

	public short getSerializeType() {
		return serializeType;
	}

    public void setProxy(boolean proxy) {
		this.proxy = proxy;
	}

	public boolean isProxy(){
    	return this.proxy;
    }
	
	public void postMessage(byte[] msg) {
    	ByteBuffer buff = ByteBuffer.wrap(msg);
    	this.postMessage(buff);
    }
    
    @Override
    protected PacketInputStream createPacketInputStream() {
        return new FramedInputStream(PACKET_META_DATA, true);
    }

    @Override
    protected PacketOutputStream createPacketOutputStream() {
        return new FramingOutputStream(PACKET_META_DATA, true);
    }


	public boolean needPing(long now) {
        /*return (((now - lastPingTime)>pingInterval) 
        && (now - lastMessageSent > pingInterval));*/
		return false;
    }

    protected void messageProcess(byte[] msg) {
    	
    	if(msg == null || msg.length < PacketConstant.SERVICE_HEADER_SIZE){
    		if(logger.isInfoEnabled()){
    			logger.info("conn ID="+this.getId()+" , received packet size error, size="+(msg == null?0: msg.length));
    		}
    		this.postClose(new IllegalArgumentException("packet size error size="+(msg == null?0: msg.length)));
    		return;
    	}
    	
        int type = AbstractServicePacket.getType(msg);
        if (type == PacketConstant.PACKET_TYPE_PONG) {
            lastPongTime = System.currentTimeMillis();
            if (logger.isDebugEnabled()) {
                logger.debug("receive pong packet from " + this.getId());
            }
        }
        super.messageProcess(msg);
    }

    public void ping(long now) {
        postMessage(new PingPacket().toByteBuffer());
        lastPingTime = System.currentTimeMillis();
        if (logger.isDebugEnabled()) {
            logger.debug("send ping packet to " + this.getId());
        }
    }

    /**
	 * ڴ֤Connection IdleʱӦһ㡣
	 */
	public boolean checkIdle(long now) {
		if (isAuthenticated()) {
			return false;
		} else {
			boolean idle = super.checkIdle(now);
			if(!idle){
				if(lastPingTime - lastPongTime > 2 * pingInterval){
					logger.warn("receive pong packet timeout, id="+this.getId()+",lastPingTime="+new Date(lastPingTime)+",lastPongTime ="+new Date(lastPongTime));
					return true;
				}
			}
			long idleMillis = now - _lastEvent;
			if (idleMillis < getAuthingTimeout()) {
				return false;
			}
			
			return true;
		}
	}

    
	protected void beforeAuthing() {
        HandshakePacket handshakePacket = new HandshakePacket();
        handshakePacket.challenge = StringUtil.getRandomString(12);
        handshakePacket.version = PacketConstant.VENUS_VERSION;
        this.putStep(1, handshakePacket);
        this.postMessage(handshakePacket.toByteBuffer());
    }
	
    protected void connectionAuthenticateSuccess(AuthResponseData data) {
    	 super.connectionAuthenticateSuccess( data);
    	 
         postMessage(OK.toByteBuffer());
    }

    protected void connectionAuthenticateFaild(AuthResponseData data) {
    	super.connectionAuthenticateFaild(data);
        ErrorPacket error = new ErrorPacket();
        error.errorCode = data.code;
        error.message = data.message;
        postMessage(error.toByteBuffer());
    }
    

	@Override
	public void handleMessage(Connection conn, byte[] message) {
		
		/** ʱյӦ֤ݣΪ֤ṩ */
		int type = AbstractServicePacket.getType(message);
		
		if(type != PacketConstant.PACKET_TYPE_AUTHEN){
			ErrorPacket error = new ErrorPacket();
			error.message = "connection must be authenticated before it send other type packet!";
			error.errorCode =VenusExceptionCodeConstant.AUTHEN_EXCEPTION;
	        postMessage(error.toByteBuffer());
		}
		
		AuthenPacket autheticationPacket;
		try {
			autheticationPacket = AuthenPacketFactory.getInstance().createAuthenPacket(message);
			
			//set connection context
			{
				this.clientId = autheticationPacket.clientId;
				this.serializeType = autheticationPacket.shakeSerializeType;
			}
			
		} catch (Exception e) {
			ErrorPacket error = new ErrorPacket();
			if(e instanceof CodedException){
				error.errorCode = ((CodedException)e).getErrorCode();
			}else{
				error.errorCode = VenusExceptionCodeConstant.SERVICE_UNAVAILABLE_EXCEPTION;
			}
	        
	        error.message = e.getMessage();
	        postMessage(error.toByteBuffer());
	        return;
		}

		this.getAuthenticateProvider().authenticateConnection(this,autheticationPacket);
	}
}
