/**
 * <pre>
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU AFFERO GENERAL PUBLIC LICENSE as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU AFFERO GENERAL PUBLIC LICENSE for more details. 
 * 	You should have received a copy of the GNU AFFERO GENERAL PUBLIC LICENSE along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * </pre>
 */
package com.meidusa.toolkit.net;

import java.io.IOException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Map.Entry;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.LinkedBlockingQueue;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.meidusa.toolkit.net.buffer.BufferPool;
import com.meidusa.toolkit.net.config.ExceptionCodeConstant;
import com.meidusa.toolkit.net.util.LoopingThread;
import com.meidusa.toolkit.util.ExecutorUtil;
import com.meidusa.toolkit.util.NameableExecutor;

/**
 * 连接管理器
 * 
 */
public class ConnectionManager extends LoopingThread{
    protected static final int                       CONNECTION_ESTABLISHED          = 0;

    protected static final int                       CONNECTION_FAILED               = 1;
    protected static final int                       CONNECTION_CLOSED               = 2;
    protected static final int                       CONNECTION_AUTHENTICATE_SUCCESS = 3;
    protected static final int                       CONNECTION_AUTHENTICATE_FAILD   = 4;
	private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionManager.class);
	private static final Logger REPORT = LoggerFactory.getLogger("venus.report.queue");
    private static final List<ConnectionManager> MANAGERS = new ArrayList<ConnectionManager>(); 
    protected List<ConnectionObserver>	_observers	= new ArrayList<ConnectionObserver>();
    
    private static class IdleChecker extends LoopingThread{
    	
    	IdleChecker(){
    		this.setName("Connection Idle Checker");
    		this.setDaemon(true);
    	}
    	
    	public void iterate(){
    		try {
				Thread.sleep(5000);
			} catch (InterruptedException e) {
			}
    		int queueSize = 0;
    		for(ConnectionManager manager:MANAGERS){
    			manager.connectionsCheck();
    			if(manager.getExecutor() != null){
    			    queueSize = +manager.getExecutor().getQueue().size();
    			}
    		}
    		
    		REPORT.info("server total queue="+queueSize);
    	}
    	
    	protected void handleIterateFailure(Throwable e) {
    		LOGGER.warn("Idle checker running exception", e);
    	}
    }
    static {
    	new IdleChecker().start();
    }

    private final Selector selector;
    private final BlockingQueue<Connection> registerQueue;
    private final ConcurrentMap<Long, AbstractConnection> frontends;
    private final ConcurrentMap<Long, AbstractConnection> backends;
    private final NameableExecutor executor;
    private final BufferPool bufferPool;
    private long netInBytes;
    private long netOutBytes;
    
    public ConnectionManager(String name,int executorSize) throws IOException {
        this.setName(name);
        this.bufferPool = new BufferPool(16 * 1024 * 1024, 8 * 1024);
        this.selector = Selector.open();
        this.registerQueue = new LinkedBlockingQueue<Connection>();
        this.frontends = new ConcurrentHashMap<Long, AbstractConnection>();
        this.backends = new ConcurrentHashMap<Long, AbstractConnection>();
        this.executor = (executorSize > 0) ? ExecutorUtil.create(name + "-Eexecutor", executorSize) : null;
        MANAGERS.add(this);
    }

    protected void willStart() {
    	super.willStart();
    }
    
    public long getNetInBytes() {
        return netInBytes;
    }

    public void addNetInBytes(long bytes) {
        netInBytes += bytes;
    }

    public long getNetOutBytes() {
        return netOutBytes;
    }

    public void addNetOutBytes(long bytes) {
        netOutBytes += bytes;
    }
    
    public BufferPool getBufferPool() {
		return bufferPool;
	}

	final void postRegister(Connection c) {
        registerQueue.offer(c);
        selector.wakeup();
    }

    final BlockingQueue<Connection> getRegisterQueue() {
        return registerQueue;
    }

    public NameableExecutor getExecutor() {
		return executor;
	}

    public void addConnection(AbstractConnection c) {
    	if(c instanceof FrontendConnection){
    		frontends.put(c.getId(), c);
    	}else{
    		backends.put(c.getId(), c);
    	}
    }

    @Override
    public void iterate() throws IOException {
        selector.select(1000L);
        register(selector);
        Set<SelectionKey> keys = selector.selectedKeys();
        try {
            for (SelectionKey key : keys) {
                Object att = key.attachment();
                if (att != null && key.isValid()) {
                    int readyOps = key.readyOps();
                    if ((readyOps & SelectionKey.OP_READ) != 0) {
                        read((Connection) att);
                    } else if ((readyOps & SelectionKey.OP_WRITE) != 0) {
                        write((Connection) att);
                    } else {
                        key.cancel();
                    }
                } else {
                    key.cancel();
                }
            }
        } finally {
            keys.clear();
        }
    }

    private void register(Selector selector) {
        Connection c = null;
        while ((c = registerQueue.poll()) != null) {
            try {
                c.register(selector);
                if(!c.isClosed()){
                	this.notifyObservers(CONNECTION_ESTABLISHED, c, null);
                }
            } catch (Throwable e) {
                c.handleError(ExceptionCodeConstant.ERR_REGISTER, e);
            }
        }
    }

    private void read(Connection c) {
        try {
            c.read();
        } catch (Throwable e) {
            c.handleError(ExceptionCodeConstant.ERR_READ, e);
        }
    }

    private void write(Connection c) {
        try {
            c.writeByEvent();
        } catch (Throwable e) {
            c.handleError(ExceptionCodeConstant.ERR_WRITE_BY_EVENT, e);
        }
    }
    
    public int getFrontends(){
    	return this.frontends.size();
    }
    
    public int getBackends(){
    	return this.backends.size();
    }
    
    // 前端连接检查
    private void connectionsCheck() {
        Iterator<Entry<Long, AbstractConnection>> it = frontends.entrySet().iterator();
        while (it.hasNext()) {
        	AbstractConnection c = it.next().getValue();

            // 删除空连接
            if (c == null) {
                it.remove();
                continue;
            }

            // 清理已关闭连接，否则空闲检查。
            if (c.isClosed()) {
                it.remove();
                c.cleanup();
            } else {
                c.idleCheck();
            }
        }
        
         it = backends.entrySet().iterator();
        while (it.hasNext()) {
        	AbstractConnection c = it.next().getValue();

            // 删除空连接
            if (c == null) {
                it.remove();
                continue;
            }

            // 清理已关闭连接，否则空闲检查。
            if (c.isClosed()) {
                it.remove();
                c.cleanup();
            } else {
                c.idleCheck();
            }
        }
    }
    
    /**
     * 增加 ConnectionObserver。监听Connection 相关的网络事件
     */
    public void addConnectionObserver(ConnectionObserver observer) {
        _observers.add(observer);
    }

    /**
     * 从 Observer 列表中删除一个Observer对象
     */
    public void removeConnectionObserver(ConnectionObserver observer) {
        _observers.remove(observer);
    }
    

    protected void notifyObservers(int code, Connection conn, Object arg1) {
        for (ConnectionObserver obs : _observers) {
            switch (code) {
                case CONNECTION_ESTABLISHED:
                    obs.connectionEstablished(conn);
                    break;
                case CONNECTION_FAILED:
                    obs.connectionFailed(conn, (Exception) arg1);
                    break;
                case CONNECTION_CLOSED:
                    obs.connectionClosed(conn);
                    break;
                default:
                    throw new RuntimeException("Invalid code supplied to notifyObservers: " + code);
            }
        }
    }

}
