/**
 * <pre>
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU General Public License for more details. 
 * 	You should have received a copy of the GNU General Public License along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * </pre>
 */
package com.meidusa.toolkit.net.packet;

import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;

import com.meidusa.toolkit.common.util.CharsetCache;
import com.meidusa.toolkit.net.IllegalPacketException;

/**
 * 该类负责 发送、接收socket输入流，并且可以根据包头信息，构造出ByteBuffer 该类采用网络字节序进行通讯
 * 
 * @author <a href=mailto:piratebase@sina.com>Struct chen</a>
 */
public abstract class GenericIOPacketBuffer extends AbstractPacketBuffer {
	public abstract int getHeadSize();
	public GenericIOPacketBuffer(byte[] buf) {
		super(buf);
	}

	public GenericIOPacketBuffer(int size) {
		super(size);
	}

	public final byte[] getBytes(int len) {
		byte[] b = new byte[len];
		this.readBytes(b);
		return b;
	}

	public byte[] getBytes(int offset, int len) {
		byte[] dest = new byte[len];
		System.arraycopy(this.data, offset, dest, 0, len);
		return dest;
	}

	/**
	 * 表示后面所要读取内容的长度
	 * 
	 * @return
	 */
	public abstract int readFieldLength();

	public final int readInt() {
		return ((data[position++] & 0xff) << 24) |
		((data[position++] & 0xff) << 16) |
		((data[position++] & 0xff) <<  8) |
		((data[position++] & 0xff) <<  0);
	}

	public final void writeInt(int i) {
		ensureCapacity(4);
		this.position = writeInt(this.data,this.position,i);
	}
	
	public static final int readInt(byte[] data,int position){
		return ((data[position++] & 0xff) << 24) |
		((data[position++] & 0xff) << 16) |
		((data[position++] & 0xff) <<  8) |
		((data[position++] & 0xff) <<  0);
	}
	
	public static final int writeInt(byte[] data,int position,int i) {
		data[position++] = (byte)(i >> 24);
		data[position++] = (byte)(i >> 16);
		data[position++] = (byte)(i >> 8);
		data[position++] = (byte)(i >> 0);
		return position;
	}
	
	public final long readLong() {
		return (((((long)data[position++]) & 0xff) << 56) |
				((((long)data[position++]) & 0xff) << 48) |
				((((long)data[position++]) & 0xff) << 40) |
				((((long)data[position++]) & 0xff) << 32) |
				((((long)data[position++]) & 0xff) << 24) |
				((((long)data[position++]) & 0xff) << 16) |
				((((long)data[position++]) & 0xff) <<  8) |
				((((long)data[position++]) & 0xff) <<  0));
	}
	
	public static final long readLong(byte[] data,int position) {
		return (((((long)data[position++]) & 0xff) << 56) |
				((((long)data[position++]) & 0xff) << 48) |
				((((long)data[position++]) & 0xff) << 40) |
				((((long)data[position++]) & 0xff) << 32) |
				((((long)data[position++]) & 0xff) << 24) |
				((((long)data[position++]) & 0xff) << 16) |
				((((long)data[position++]) & 0xff) <<  8) |
				((((long)data[position++]) & 0xff) <<  0));
	}

	public final void writeLong(long i) {
		ensureCapacity(8);
		this.position = writeLong(this.data,position,i);
	}
	
	public static final int writeLong(byte[] data ,int position,long i) {
		data[position++] = (byte)(i >> 56);
		data[position++] = (byte)(i >> 48);
		data[position++] = (byte)(i >> 40);
		data[position++] = (byte)(i >> 32);
		data[position++] = (byte)(i >> 24);
		data[position++] = (byte)(i >> 16);
		data[position++] = (byte)(i >> 8);
		data[position++] = (byte)(i >> 0);
		return position;
	}
	
	public int writeLengthCodedBytes(byte[] ab) {
		if(ab==null){
			ensureCapacity(4);
		}else{
			ensureCapacity(ab.length+4);
		}
    	this.writeInt(ab==null?0:ab.length);
    	if(ab != null && ab.length > 0){
    		return writeBytes(ab, 0, ab.length);
    	}else{
    		return 0;
    	}
    }
	
	public void skipLengthCodedBytes() {
		int count = readInt();
		if(count ==0){
			return;
		}else{
			if(count > this.remaining()){
				throw new IllegalPacketException("packet content size error: "+count +">" +this.remaining()+" remaining");
			}
			this.skip(count);
		}
	}
	public byte[] readLengthCodedBytes() {
		int count = readInt();
		if(count ==0){
			return new byte[0];
		}else{
			if(count > this.remaining()){
				throw new IllegalPacketException("packet content size error: "+count +">" +this.remaining()+" remaining");
			}
			byte[] bts = new byte[count];
			this.readBytes(bts, 0, count);
			return bts;
		}
    }
	
	public final String readLengthCodedString(Charset charset) {
		int fieldLength = (int) readFieldLength();
		
		if (fieldLength == 0) {
			return null;
		}
		
		if(fieldLength> length-position){
			throw new IllegalPacketException("fieldLength error Buffer.Remaining="+remaining()+" ,but need size="+ fieldLength);
		}
		byte[] bytes = getBytes(fieldLength);
		if (charset != null) {
			return new String(bytes,charset);
		} else {
			return new String(bytes);
		}
	}
	
	public final String readLengthCodedString(String encoding) {
		Charset charset = CharsetCache.getCharset(encoding);
		return readLengthCodedString(charset);
	}

	public String toSuperString() {
		return super.toString();
	}

	public abstract void writeFieldLength(int length);

    public short readShort(){
    	return (short)(((this.data[position++] & 0xff) << 8) | (this.data[position++] & 0xff));
    }
    
    public void writeShort(short i) {
        ensureCapacity(2);
        this.position = writeShort(this.data,this.position,i);
    }
    
    public static final int writeShort(byte[] data,int position,short i) {
		data[position++] = (byte)(i >> 8);
		data[position++] = (byte)(i >> 0);
		return position;
    }
    
	public final void writeLengthCodedString(String s, String encoding) {
		Charset charset = CharsetCache.getCharset(encoding);
		writeLengthCodedString(s,charset);
	}

	public final void writeLengthCodedString(String s, Charset charset) {
		if (s != null) {
			byte[] b;
			b = s.getBytes(charset);
			ensureCapacity(b.length + 9);
			this.writeFieldLength(b.length);
			this.writeBytes(b);
		} else {
			this.writeFieldLength(0);
		}
	}
	
	// Write null-terminated string
	public final void writeString(String s) {
		try {
			writeString(s, null);
		} catch (UnsupportedEncodingException e) {
		}
	}

	public final void writeString(String s, String encoding)
			throws UnsupportedEncodingException {
		byte[] bytes = null;
		if (encoding == null) {
			bytes = s.getBytes();
		} else {
			bytes = s.getBytes(encoding);
		}
		ensureCapacity(bytes.length + 1 + 8);
		this.writeFieldLength(bytes.length);
		this.writeBytes(bytes);
	}

}
