package com.meidusa.toolkit.net.packet;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import com.meidusa.toolkit.net.buffer.BufferPool;

/**
 * @author struct
 */
public class AbstractPacketBuffer implements PacketBuffer {
	
	public static final BufferPool BUFFER_POOL = new BufferPool(64 * 1024 * 1024, 8 * 1024);
    protected byte[] data;
    protected int length;
    protected int position;
    
    public AbstractPacketBuffer(byte[] data){
    	this.data = data;
    	this.length = data.length;
    }

    public AbstractPacketBuffer(int size){
    	data = new byte[size];
    }

    public ByteOrder getByteOrder(){
    	return ByteOrder.BIG_ENDIAN;
    }
    /**
     * 将从0到当前位置的所有字节写入到ByteBuffer中,写完以后将位置设置到以前位置
     */
    public ByteBuffer toByteBuffer() {
    	/*byte[] bts = new byte[position];
    	System.arraycopy(this.data, 0, bts, 0, position);*/
    	ByteBuffer buffer = null;
    	if(position <= AbstractPacketBuffer.BUFFER_POOL.getChunkSize()){
    		buffer = AbstractPacketBuffer.BUFFER_POOL.allocate();
    		buffer.put(this.data,0,position);
    		buffer.flip();
    	}else{
    		buffer = ByteBuffer.wrap(this.data,0,position);
    	}
        //ByteBuffer buffer = ByteBuffer.wrap(bts);
        //buffer.rewind();
        return buffer;
    }

    public final static ByteBuffer toByteBuffer(byte[]data,int offset,int lenght) {
    	ByteBuffer buffer = null;
    	if(lenght <= AbstractPacketBuffer.BUFFER_POOL.getChunkSize()){
    		buffer = AbstractPacketBuffer.BUFFER_POOL.allocate();
    		buffer.put(data,offset,lenght);
    		buffer.flip();
    	}else{
    		buffer = ByteBuffer.wrap(data,offset,lenght);
    	}
        return buffer;
    }
    
    
    public int getPosition() {
        return position;
    }

    public void setPosition(int position) {
        this.position = position;
    }

    public byte readByte() {
    	return data[position++];
    }

    public byte readByte(int i) {
    	return data[i];
    }

    public void writeByte(byte b) {
        ensureCapacity(1);
        data[position++] = b;
    }
    
    public int writeBytes(byte[] ab) {
        return writeBytes(ab, 0, ab.length);
    }
    
    public void readBytes(byte[] ab) {
    	readBytes(ab,0,ab.length);
    }

    public int writeBytes(byte[] ab, int offset, int len) {
        ensureCapacity(len);
        System.arraycopy(ab, offset, this.data, this.position, len);
        this.position += len;
        return len;
    }

    public int readBytes(byte[] ab, int offset, int len) {
    	System.arraycopy(this.data, this.position, ab, offset, len);
   	 	this.position += len;
        return len;
    }

    /**
     * 增加buffer长度
     */
    protected void ensureCapacity(int i) {
        if (this.data.length-this.position <i) {
        	int size = Math.max(this.data.length << 1, i+position);
        	
        	byte[] newData = new byte[size];
        	System.arraycopy(this.data, 0, newData, 0, this.position);
        	this.data = newData;
        }
    }

    protected void init() {
    }

    public synchronized void reset() {
        this.position = 0;
    }

    public byte[] readRemaining(){
    	byte[] bts = new byte[data.length-position];
    	System.arraycopy(data, this.position, bts, 0, bts.length);
		return bts;
    }
    public int remaining() {
        return this.data.length - this.position;
    }

    public boolean hasRemaining() {
        return this.data.length - this.position >0;
    }

    public void skip(int bytes) {
    	this.position +=bytes;
    }

    public InputStream asInputStream() {
        return new InputStream() {

            @Override
            public int available() {
                return AbstractPacketBuffer.this.remaining();
            }

            @Override
            public int read() {
                if (AbstractPacketBuffer.this.hasRemaining()) {
                    return AbstractPacketBuffer.this.readByte() & 0xff;
                } else {
                    return -1;
                }
            }

            @Override
            public int read(byte[] b, int off, int len) {
                int remaining = AbstractPacketBuffer.this.remaining();
                if (remaining > 0) {
                    int readBytes = Math.min(remaining, len);
                    AbstractPacketBuffer.this.readBytes(b, off, readBytes);
                    return readBytes;
                } else {
                    return -1;
                }
            }

            @Override
            public synchronized void reset() {
                AbstractPacketBuffer.this.reset();
            }

            @Override
            public long skip(long n) {
                int bytes;
                if (n > Integer.MAX_VALUE) {
                    bytes = AbstractPacketBuffer.this.remaining();
                } else {
                    bytes = Math.min(AbstractPacketBuffer.this.remaining(), (int) n);
                }
                AbstractPacketBuffer.this.skip(bytes);
                return bytes;
            }
        };
    }

    public OutputStream asOutputStream() {
        return new OutputStream() {

            @Override
            public void write(byte[] b, int off, int len) {
                AbstractPacketBuffer.this.writeBytes(b, off, len);
            }

            @Override
            public void write(int b) {
                AbstractPacketBuffer.this.writeByte((byte) b);
            }
        };
    }
    
    /**
	 * 往buffer中写入固定长度的字节。如果字符串长度不足则补足长度。超过将被截。
	 * @param buffer
	 * @param string 被写入的字符串
	 * @param encoding 采用的编码
	 * @param length 写入固定长度
	 */
	public void writeFixedLengthString(String string,String encoding,int length){
		if(string == null){
			writeBytes(new byte[length]);
		}else{
			byte[] strBytes = null;
			if(encoding != null){
				try {
					strBytes = string.getBytes(encoding);
				} catch (UnsupportedEncodingException e) {
					strBytes = string.getBytes();
				}
			}else{
				strBytes = string.getBytes();
			}

			if(length<=strBytes.length){
				writeBytes(strBytes,0,length);
			}else{
				writeBytes(strBytes);
				writeBytes(new byte[length-strBytes.length]);
			}
		}
	}
	
	public String readFixedLengthString(String encoding,int length){
		if(length > this.remaining() ){
			throw new ArrayIndexOutOfBoundsException("packet remaining="+this.remaining()+" but need size="+length);
		}
		byte[] strBytes = new byte[length];
    	readBytes(strBytes,0,length);
    	int nullIndex = -1;
    	int noNullLength = 0;
    	for(int i=strBytes.length-1;i>=0;i--){
    		if(strBytes[i] != (byte)0){
    			nullIndex = i;
    			break;
    		}
    	}
    	
    	if(nullIndex == 0){
    		return null;
    	}
    	
    	if(nullIndex == -1){
    		noNullLength = length;
    	}else{
    		noNullLength = nullIndex+1;
    	}
    	
    	if(encoding == null){
    		return new String(strBytes,0,noNullLength);
    	}else{
    		try {
				return new String(strBytes,0,noNullLength, encoding);
			} catch (UnsupportedEncodingException e) {
				return new String(strBytes,0,noNullLength);
			}
    	}
	}
	
	public static void main(String[] args){
		byte[] byts = new byte[]{(byte)1,(byte)0,(byte)0,(byte)0,(byte)12,(byte)1,(byte)0,(byte)0};
		AbstractPacketBuffer buffer = new AbstractPacketBuffer(byts);
		String ms = buffer.readFixedLengthString(null, 7);
		System.out.println(ms);
	}

}
