/**
 * <pre>
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU AFFERO GENERAL PUBLIC LICENSE as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU AFFERO GENERAL PUBLIC LICENSE for more details. 
 * 	You should have received a copy of the GNU AFFERO GENERAL PUBLIC LICENSE along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * </pre>
 */
package com.meidusa.toolkit.net.factory;

import java.io.IOException;
import java.net.Socket;
import java.nio.channels.SocketChannel;

import com.meidusa.toolkit.net.BackendConnection;
import com.meidusa.toolkit.net.ConnectionConnector;
import com.meidusa.toolkit.net.ValidatorMessageHandler;
import com.meidusa.toolkit.net.buffer.BufferQueue;

public abstract class BackendConnectionFactory {

    protected int receiveBufferSize = 16;
    protected int sendBufferSize = 8;
    protected int maxPacketSize = 16 * 1024 * 1024;
    protected int writeQueueCapcity = 8;
    protected long idleTimeout = 8 * 3600 * 1000L;
    private ConnectionConnector connector;
    protected String host;
    protected int port;
    
    public abstract ValidatorMessageHandler createValidatorMessageHandler() ;

	public ConnectionConnector getConnector() {
		return connector;
	}

	public void setConnector(ConnectionConnector connector) {
		this.connector = connector;
	}

	public BackendConnection make() throws IOException {
        SocketChannel channel = openSocketChannel();
        BackendConnection c = create(channel);
        postConnect(c, getConnector());
        return c;
    }
	
	protected abstract BackendConnection create(SocketChannel channel);
    
    protected SocketChannel openSocketChannel() throws IOException {
        SocketChannel channel = null;
        try {
            channel = SocketChannel.open();
            channel.configureBlocking(false);
            Socket socket = channel.socket();
            socket.setReceiveBufferSize(receiveBufferSize * 1024);
            socket.setSendBufferSize(sendBufferSize * 1024);
            socket.setTcpNoDelay(true);
            socket.setKeepAlive(true);
        } catch (IOException e) {
            closeChannel(channel);
            throw e;
        } catch (RuntimeException e) {
            closeChannel(channel);
            throw e;
        }
        return channel;
    }

    protected void postConnect(BackendConnection c, ConnectionConnector connector) {
        c.setMaxPacketSize(maxPacketSize);
        c.setWriteQueue(new BufferQueue(writeQueueCapcity));
        c.setIdleTimeout(idleTimeout);
        c.setHost(host);
        c.setPort(port);
        c.setConnector(connector);
        connector.postConnect(c);
    }

    public int getReceiveBufferSize() {
        return receiveBufferSize;
    }

    public void setReceiveBufferSize(int socketRecvBuffer) {
        this.receiveBufferSize = socketRecvBuffer;
    }

    public int getSendBufferSize() {
        return sendBufferSize;
    }

    public void setSendBufferSize(int socketSendBuffer) {
        this.sendBufferSize = socketSendBuffer;
    }

    public int getMaxPacketSize() {
        return maxPacketSize;
    }

    public void setMaxPacketSize(int maxPacketSize) {
        this.maxPacketSize = maxPacketSize;
    }

    public int getWriteQueueCapcity() {
        return writeQueueCapcity;
    }

    public void setWriteQueueCapcity(int writeQueueCapcity) {
        this.writeQueueCapcity = writeQueueCapcity;
    }

    public long getIdleTimeout() {
        return idleTimeout;
    }

    public void setIdleTimeout(long idleTimeout) {
        this.idleTimeout = idleTimeout;
    }

    public String getHost() {
		return host;
	}

	public void setHost(String host) {
		this.host = host;
	}

	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	private static void closeChannel(SocketChannel channel) {
        if (channel == null) {
            return;
        }
        Socket socket = channel.socket();
        if (socket != null) {
            try {
                socket.close();
            } catch (IOException e) {
            }
        }
        try {
            channel.close();
        } catch (IOException e) {
        }
    }

}
