package com.meidusa.toolkit.net;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


import com.meidusa.toolkit.common.heartbeat.HeartbeatManager;
import com.meidusa.toolkit.net.Connection;
import com.meidusa.toolkit.net.factory.BackendConnectionFactory;

public class SimpleBackendConnectionPool<F extends BackendConnectionFactory, V extends BackendConnection>
		implements BackendConnectionPool {
	private static final Logger LOGGER = LoggerFactory.getLogger(SimpleBackendConnectionPool.class);

	private final ReentrantLock lock = new ReentrantLock();
	private final F factory;
	private final int size;

	private final BackendConnection[] items;
	private int activeCount;
	private int idleCount;
	private String name;
	private boolean valid = true;
	private boolean closed = false;

	@Override
	public String getName() {
		return name;
	}

	public SimpleBackendConnectionPool(String name, F factory, int size) {
		this.size = size;
		this.items = new BackendConnection[size];
		this.factory = factory;
		this.name = name;
	}

	@Override
	public BackendConnection borrowObject() throws Exception {
		final ReentrantLock lock = this.lock;
		lock.lock();
		try {
			// too many active connections
			if (activeCount >= size) {
				StringBuilder s = new StringBuilder();
				s.append("[name=").append(name).append(",active=");
				s.append(activeCount).append(",size=").append(size).append(']');
				LOGGER.error(s.toString());
			}

			// get connection from pool
			final BackendConnection[] items = this.items;
			for (int i = 0, len = items.length; idleCount > 0 && i < len; ++i) {
				if (items[i] != null) {
					BackendConnection conn = items[i];
					items[i] = null;
					--idleCount;
					if (conn.isClosed()) {
						continue;
					} else {
						++activeCount;
						conn.setActive(true);
						return conn;
					}
				}
			}

			++activeCount;
		} finally {
			lock.unlock();
		}

		// create connection
		try {
			BackendConnection conn = factory.make();
			conn.setPool(this);
			conn.setActive(true);
			return conn;
		} catch (Exception e) {
			lock.lock();
			try {
				--activeCount;
			} finally {
				lock.unlock();
			}
			throw e;
		}
	}

	@Override
	public void returnObject(BackendConnection c) {
		if (c == null || c.isClosed()) {
			return;
		}
		
		if(c instanceof AuthingableBackendConnection){
			if(!((AuthingableBackendConnection) c).isAuthenticated()){
				return;
			}
		}
		
		// release connection
		final ReentrantLock lock = this.lock;
		lock.lock();
		try {
			final Connection[] items = this.items;
			for (int i = 0; i < items.length; i++) {
				if (items[i] == null) {
					++idleCount;
					--activeCount;
					items[i] = c;
					c.setActive(false);
					return;
				}
			}
		} finally {
			lock.unlock();
		}
		// close excess connection
		c.close();
	}

	@Override
	public synchronized void close() {
		if(closed){
			return;
		}
		closed = true;
		HeartbeatManager.removeHeartbeat(delay);
		final ReentrantLock lock = this.lock;
		lock.lock();
		try {
			for (int i = 0, len = items.length; idleCount > 0 && i < len; ++i) {
				if (items[i] != null) {
					Connection conn = items[i];
					items[i] = null;
					--idleCount;
					if (conn.isClosed()) {
						continue;
					}
					conn.close();
				}
			}
		} finally {
			lock.unlock();
		}
	}


	@Override
	public int getActive() {
		final ReentrantLock lock = this.lock;
		lock.lock();
		try {
			return activeCount;
		} finally {
			lock.unlock();
		}
	}

	@Override
	public void deActive(BackendConnection c) {
		final ReentrantLock lock = this.lock;
		lock.lock();
		try {
			--activeCount;
		} finally {
			lock.unlock();
		}
	}

	@Override
	public boolean isValid() {
		return valid;
	}

	@Override
	public void setValid(boolean b) {
		this.valid = b;
	}

	private GenericHeartbeatDelayed delay = new GenericHeartbeatDelayed(HeartbeatManager.DEFAULT_HEATBEAT_INTERVAL,TimeUnit.SECONDS, this);

	public void init() {
		HeartbeatManager.addHeartbeat(delay);
	}

	@Override
	public boolean isClosed() {
		return closed;
	}
	
	
}
