package com.meidusa.toolkit.net;

import java.io.IOException;
import java.io.InvalidObjectException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.meidusa.toolkit.common.heartbeat.HeartbeatManager;
import com.meidusa.toolkit.common.heartbeat.Status;
import com.meidusa.toolkit.net.factory.BackendConnectionFactory;

public class PollingBackendConnectionPool <F extends BackendConnectionFactory, V extends BackendConnection> implements BackendConnectionPool {
	private static final Logger LOGGER = LoggerFactory.getLogger(PollingBackendConnectionPool.class);
	private static int HEATBEAT_INTERVAL = Integer.getInteger("heartbeat.interval", HeartbeatManager.DEFAULT_HEATBEAT_INTERVAL);
	/**
	 * 非严格的计数器
	 */
	private volatile long currentCounter = 0L;
	private final F factory;
	private final int size;
	private final BackendConnection[] items;
	private String name;
	private boolean valid = true;
	private boolean closed = false;
	private final Map<Integer,Object> lockMap = new HashMap<Integer,Object>();
	public PollingBackendConnectionPool(String name, F factory, int size) {
		this.size = size;
		this.items = new BackendConnection[size];
		this.factory = factory;
		this.name = name;
	}
	
	@Override
	public String getName() {
		return name;
	}

	@Override
	public BackendConnection borrowObject() throws Exception {
		BackendConnection conn = null;
		
		/**
		 * 循环次数为 size+1，主要目的是避免目标服务重启以后，当前的所有连接已经处于closed状态，
		 * 必须要再进行创建一次连接，如果该连接有效则说明pool有效，否则视为无效
		 * 
		 */
		for(int j=0;j<size+1;j++){
			int i = (int)((currentCounter++) % size);
			conn = items[i] ;
			if(conn == null){
				Object lockObject = lockMap.get(i);
				synchronized (lockObject) {
					conn = items[i];
					if(conn == null){
						conn = items[i] = factory.make();
						conn.setPool(this);
					}
				}
			}
			
			if(!conn.isClosed()){
				conn.setActive(true);
				return conn;
			}else{
				items[i] = null;
			}
		}
		
		this.setValid(false);
		throw new InvalidObjectException("invalid pool="+this.name);
	}

	@Override
	public void returnObject(BackendConnection c) {
		c.setActive(false);
	}

	@Override
	public synchronized void close() {
		if(closed){
			return ;
		}
		closed = true;
		if(delay != null){
			HeartbeatManager.removeHeartbeat(delay);
		}
		for(BackendConnection conn : items){
			if(conn != null && !conn.isClosed()){
				conn.close();
			}
		}
	}

	@Override
	public int getActive() {
		return items.length;
	}

	@Override
	public void deActive(BackendConnection c) {
		//polling属于循环，该pool中连接可以重用，因此无所谓是否被使用
		c.setActive(false);
	}

	@Override
	public boolean isValid() {
		return this.valid;
	}

	@Override
	public void setValid(boolean b) {
		this.valid = b;
	}

	public boolean isClosed(){
		return closed;
	}
	
	private GenericHeartbeatDelayed delay;
	
	@Override
	public void init() {
		delay = new GenericHeartbeatDelayed(HEATBEAT_INTERVAL,TimeUnit.SECONDS, this){
			Status last = Status.VALID;
			public Status doCheck() {
				
				Status status = check0();
				try{
					if (last == Status.INVALID && status == Status.VALID) {
						//do init pool
						for (int i = 0; i < size; i++) {
							status = check0();
							if (status == Status.INVALID) {
								return status;
							}
						}
						return status;
					}
					
					return status;
				}finally{
					last = status;
				}
			}

			private Status check0() {
				BackendConnection conn = null;
				try {
					conn = pool.borrowObject();
					if (conn != null) {

						if (conn.isClosed()) {
							pool.setValid(false);
							return Status.INVALID;
						} else {
							BackendConnection other = null;
							ValidatorMessageHandler handler = factory.createValidatorMessageHandler();
							if(handler != null){
								try{
									other = factory.make();
									if(!other.isClosed()){
										other.setHandler(handler);
										handler.doCheck(other);
										Status status = handler.getResult(conn);
										if(status == Status.VALID){
											pool.setValid(true);
											return Status.VALID;
										}else{
											pool.setValid(false);
											return status;
										}
									}
									
								}finally{
									if(other != null){
										other.close();
									}
								}
							}
							pool.setValid(true);
							return Status.VALID;
						}
					} else {
						return Status.INVALID;
					}
				} catch (Exception e) {
					logger.warn(e.getMessage(), e);
					pool.setValid(false);
					return Status.INVALID;
				} finally {
					if (conn != null) {
						pool.returnObject(conn);
					}
				}
			}
		};
		
		for(int i=0;i<items.length;i++){
			this.lockMap.put(i, new Object());
		}
		for(int i=0;i<items.length;i++){
			try {
				items[i] = factory.make();
			} catch (IOException e) {
				LOGGER.error("init pool error ,name="+this.name,e);
				this.setValid(false);
				break;
			}
		}
		HeartbeatManager.addHeartbeat(delay);
	}
}
