package com.meidusa.toolkit.net.util;

public class BlockingMemoryQueue<T> extends MemoryQueue<T> implements BlockingQueue<T> {

	public synchronized T getNonBlocking() {
		try{
			return super.getNonBlocking();
		}finally{
			this.notify();
		}
	}
	
	public synchronized void append(T item) {
		// only notify if the queue was previously empty
		append0(item, _count == 0);
	}
	
	/**
	 * Appends an item to the queue without notifying anyone. Useful for
	 * appending a bunch of items and then waking up the listener.
	 */
	public synchronized void appendSilent(T item) {
		append0(item, false);
	}

	/* (non-Javadoc)
	 * @see com.meidusa.toolkit.net.util.Queue#prepend(T)
	 */
	public synchronized void prepend(T item) {
		super.prepend(item);
		if (_count == 1) {
			notify();
		}
	}
	/**
	 * Appends an item to the queue and notify a listener regardless of how many
	 * items are on the queue. Use this for the last item you append to a queue
	 * in a batch via <code>appendSilent</code> because the regular
	 * <code>append</code> will think it doesn't need to notify anyone because
	 * the queue size isn't zero prior to this add. You should also use this
	 * method if you have mutiple consumers listening waiting on the queue, to
	 * guarantee that one will be woken for every element added.
	 */
	public synchronized void appendLoud(T item) {
		append0(item, true);
	}
	
	/**
	 * Blocks the current thread waiting for an item to be added to the queue.
	 * If the queue is currently non-empty, this function will return
	 * immediately.
	 */
	public synchronized void waitForItem() {
		while (_count == 0) {
			try {
				wait();
			} catch (InterruptedException e) {
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.meidusa.toolkit.net.util.Queue#get(long)
	 */
	public synchronized T get(long maxwait) {
		if (_count == 0) {
			try {
				wait(maxwait);
			} catch (InterruptedException e) {
			}

			// if count's still null when we pull out, we waited
			// ourmaxwait time.
			if (_count == 0) {
				return null;
			}
		}

		return get();
	}

	/* (non-Javadoc)
	 * @see com.meidusa.toolkit.net.util.Queue#get()
	 */
	public synchronized T get() {
		while (_count == 0) {
			try {
				wait();
			} catch (InterruptedException e) {
			}
		}

		// pull the object off, and clear our reference to it
		T retval = _items[_start];
		_items[_start] = null;

		_start = (_start + 1) % _size;
		boolean needNotify = (_count >= maxSize) && (maxSize > 0);
		_count--;

		// if we are only filling 1/8th of the space, shrink by half
		if ((_size > MIN_SHRINK_SIZE) && (_size > _suggestedSize)
				&& (_count < (_size >> 3)))
			shrink();
		if(needNotify){
			notify();
		}
		return retval;
	}
}
