/*
 * Copyright 1999-2101 Alibaba Group.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.meidusa.fastjson.parser;

import static com.meidusa.fastjson.parser.JSONScanner.EOI;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.meidusa.fastjson.JSONArray;
import com.meidusa.fastjson.JSONException;
import com.meidusa.fastjson.JSONObject;
import com.meidusa.fastjson.parser.deserializer.DefaultObjectDeserializer;
import com.meidusa.fastjson.parser.deserializer.IntegerDeserializer;
import com.meidusa.fastjson.parser.deserializer.ObjectDeserializer;
import com.meidusa.fastjson.parser.deserializer.StringDeserializer;
import com.meidusa.fastjson.util.TypeUtils;
import com.meidusa.fastmark.feature.Feature;

/**
 * @author wenshao<szujobs@hotmail.com>
 */
@SuppressWarnings("rawtypes")
public class DefaultExtJSONParser extends DefaultJSONParser {

	private DefaultObjectDeserializer derializer = new DefaultObjectDeserializer();

	private final static Map<String, Type> cachedTypes = new HashMap<String, Type>();

	public static void putType(String typeName, Type type) {
		cachedTypes.put(typeName, type);
	}

	public static Type getType(String typeName) {
		return cachedTypes.get(typeName);
	}


	public DefaultExtJSONParser(String input) {
		this(input, ParserConfig.getGlobalInstance());
	}

	public DefaultExtJSONParser(String input, ParserConfig mapping) {
		super(input, mapping);
	}

	public DefaultExtJSONParser(String input, ParserConfig mapping, int features) {
		super(input, mapping, features);
	}

	public DefaultExtJSONParser(char[] input, int length, ParserConfig mapping, int features) {
		super(input, length, mapping, features);
	}

	public ParserConfig getConfig() {
		return config;
	}

	public void setConfig(ParserConfig config) {
		this.config = config;
	}

	// compatible
	@SuppressWarnings("unchecked")
	public <T> T parseObject(Class<T> clazz) {
		return (T) parseObject((Type) clazz);
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	public Map<String, Object> parseObjectWithTypeMap(Map<String, Type> typeMap) {
	    if(typeMap == null || typeMap.size() ==0){
	       return  parseObject(JSONObject.class);
	    }
		Map<String, Object> map = new HashMap<String, Object>();
		JSONScanner lexer = (JSONScanner) this.lexer;
		if (lexer.token() != JSONToken.LBRACE) {
			throw new JSONException("syntax error, expect {, actual " + lexer.token());
		}

		for (;;) {
			lexer.skipWhitespace();
			char ch = lexer.getCurrent();
			if (isEnabled(Feature.AllowArbitraryCommas)) {
				while (ch == ',') {
					lexer.incrementBufferPosition();
					lexer.skipWhitespace();
					ch = lexer.getCurrent();
				}
			}

			String key;
			if (ch == '"') {
				key = lexer.scanSymbol(symbolTable, '"');
				lexer.skipWhitespace();
				ch = lexer.getCurrent();
				if (ch != ':') {
					throw new JSONException("expect ':' at " + lexer.pos() + ", name " + key);
				}
			} else if (ch == '}') {
				lexer.incrementBufferPosition();
				lexer.resetStringPosition();
				lexer.nextToken();
				return map;
			} else if (ch == '\'') {
				if (!isEnabled(Feature.AllowSingleQuotes)) {
					throw new JSONException("syntax error");
				}

				key = lexer.scanSymbol(symbolTable, '\'');
				lexer.skipWhitespace();
				ch = lexer.getCurrent();
				if (ch != ':') {
					throw new JSONException("expect ':' at " + lexer.pos());
				}
			} else if ((byte) ch == EOI) {
				lexer.nextToken();
				// lexer.pos();
				if (lexer.isEOF()) {
					return map;
				}
				return map;
				// throw new JSONException("syntax error");
			} else if (ch == ',') {
				throw new JSONException("syntax error");
			} else {
				if (!isEnabled(Feature.AllowUnQuotedFieldNames)) {
					throw new JSONException("syntax error");
				}

				key = lexer.scanSymbolUnQuoted(symbolTable);
				lexer.skipWhitespace();
				ch = lexer.getCurrent();
				if (ch != ':') {
					throw new JSONException("expect ':' at " + lexer.pos() + ", actual " + ch);
				}
			}
			lexer.resetStringPosition();
			lexer.nextToken();
			lexer.nextToken();
			Type type = typeMap.get(key);
			if (type == null) {
			    map.put(key, parse());
			} else {
				map.put(key, parseObject(type));
			}
			/*
			 * lexer.incrementBufferPosition(); lexer.skipWhitespace(); ch =
			 * lexer.getCurrent();
			 */

		}
	}

	@SuppressWarnings("unchecked")
	public <T> T parseObject(Type type) {
		if (lexer.token() == JSONToken.NULL) {
			lexer.nextToken();
			return null;
		}

		ObjectDeserializer derializer = config.getDeserializer(type);

		try {
			return (T) derializer.deserialze(this, type);
		} catch (JSONException e) {
			throw e;
		} catch (Throwable e) {
			throw new JSONException(e.getMessage(), e);
		}
	}

	public <T> List<T> parseArray(Class<T> clazz) {
		List<T> array = new ArrayList<T>();
		parseArray(clazz, array);
		return array;
	}

	public void parseArray(Class<?> clazz, Collection array) {
		parseArray((Type) clazz, array);
	}

	@SuppressWarnings({ "unchecked" })
	public void parseArray(Type type, Collection array) {
		int token = lexer.token();
		if (token != JSONToken.LBRACKET) {
			throw new JSONException("exepct '[', but " + JSONToken.name(lexer.token()));
		}

		ObjectDeserializer deserializer = null;
		if (int.class == type) {
			deserializer = IntegerDeserializer.instance;
			lexer.nextToken(JSONToken.LITERAL_INT);
		} else if (String.class == type) {
			deserializer = StringDeserializer.instance;
			lexer.nextToken(JSONToken.LITERAL_STRING);
		} else {
			deserializer = config.getDeserializer(type);
			lexer.nextToken(deserializer.getFastMatchToken());
		}

		for (;;) {
			if (isEnabled(Feature.AllowArbitraryCommas)) {
				while (lexer.token() == JSONToken.COMMA) {
					lexer.nextToken();
					continue;
				}
			}

			if (lexer.token() == JSONToken.RBRACKET) {
				break;
			}

			if (int.class == type) {
				Object val = IntegerDeserializer.deserialze(this);
				array.add(val);
			} else if (String.class == type) {
				String value;
				if (lexer.token() == JSONToken.LITERAL_STRING) {
					value = lexer.stringVal();
					lexer.nextToken(JSONToken.COMMA);
				} else {
					Object obj = this.parse();
					if (obj == null) {
						value = null;
					} else {
						value = obj.toString();
					}
				}

				array.add(value);
			} else {
				Object val = deserializer.deserialze(this, type);
				array.add(val);
			}

			if (lexer.token() == JSONToken.COMMA) {
				lexer.nextToken(deserializer.getFastMatchToken());
				continue;
			}
		}

		lexer.nextToken(JSONToken.COMMA);
	}

	public Object[] parseArray(Type[] types) {

		if (lexer.token() != JSONToken.LBRACKET) {
			throw new JSONException("syntax error");
		}

		Object[] list = new Object[types.length];
		if (types.length == 0) {
			lexer.nextToken(JSONToken.RBRACKET);

			if (lexer.token() != JSONToken.RBRACKET) {
				throw new JSONException("syntax error");
			}

			lexer.nextToken(JSONToken.COMMA);
			return new Object[0];
		}

		lexer.nextToken(JSONToken.LITERAL_INT);

		for (int i = 0; i < types.length; ++i) {
			Object value;

			if (lexer.token() == JSONToken.NULL) {
				value = null;
				lexer.nextToken(JSONToken.COMMA);
			} else {
				Type type = types[i];
				if (type == int.class || type == Integer.class) {
					if (lexer.token() == JSONToken.LITERAL_INT) {
						value = Integer.valueOf(lexer.intValue());
						lexer.nextToken(JSONToken.COMMA);
					} else {
						value = this.parse();
						value = TypeUtils.cast(value, type, config);
					}
				} else if (type == String.class) {
					if (lexer.token() == JSONToken.LITERAL_STRING) {
						value = lexer.stringVal();
						lexer.nextToken(JSONToken.COMMA);
					} else {
						value = this.parse();
						value = TypeUtils.cast(value, type, config);
					}
				} else {
					boolean isArray = false;
					Class<?> componentType = null;
					if (i == types.length - 1) {
						if (type instanceof Class) {
							Class<?> clazz = (Class<?>) type;
							isArray = clazz.isArray();
							componentType = clazz.getComponentType();
						}
					}

					// support varArgs
					if (isArray && lexer.token() != JSONToken.LBRACKET) {
						List<Object> varList = new ArrayList<Object>();

						ObjectDeserializer derializer = config.getDeserializer(componentType);
						int fastMatch = derializer.getFastMatchToken();

						if (lexer.token() != JSONToken.RBRACKET) {
							for (;;) {
								Object item = derializer.deserialze(this, type);
								varList.add(item);

								if (lexer.token() == JSONToken.COMMA) {
									lexer.nextToken(fastMatch);
								} else if (lexer.token() == JSONToken.RBRACKET) {
									break;
								} else {
									throw new JSONException("syntax error :" + JSONToken.name(lexer.token()));
								}
							}
						}

						value = TypeUtils.cast(varList, type, config);
					} else {
						ObjectDeserializer derializer = config.getDeserializer(type);
						value = derializer.deserialze(this, type);
					}
				}
			}
			list[i] = value;

			if (lexer.token() == JSONToken.RBRACKET) {
				break;
			}

			if (lexer.token() != JSONToken.COMMA) {
				throw new JSONException("syntax error :" + JSONToken.name(lexer.token()));
			}

			if (i == types.length - 1) {
				lexer.nextToken(JSONToken.RBRACKET);
			} else {
				lexer.nextToken(JSONToken.LITERAL_INT);
			}
		}

		if (lexer.token() != JSONToken.RBRACKET) {
			throw new JSONException("syntax error");
		}

		lexer.nextToken(JSONToken.COMMA);

		return list;
	}

	public void parseObject(Object object) {
		derializer.parseObject(this, object);
	}

	public Object parseArrayWithType(Type collectionType) {
		if (lexer.token() == JSONToken.NULL) {
			lexer.nextToken();
			return null;
		}

		Type[] actualTypes = ((ParameterizedType) collectionType).getActualTypeArguments();

		if (actualTypes.length != 1) {
			throw new JSONException("not support type " + collectionType);
		}

		Type actualTypeArgument = actualTypes[0];

		if (actualTypeArgument instanceof Class) {
			List<Object> array = new ArrayList<Object>();
			this.parseArray((Class) actualTypeArgument, array);
			return array;
		}

		if (actualTypeArgument instanceof WildcardType) {
			WildcardType wildcardType = (WildcardType) actualTypeArgument;

			// assert wildcardType.getUpperBounds().length == 1;
			Type upperBoundType = wildcardType.getUpperBounds()[0];

			// assert upperBoundType instanceof Class;
			if (Object.class.equals(upperBoundType)) {
				if (wildcardType.getLowerBounds().length == 0) {
					// Collection<?>
					return parse();
				} else {
					throw new JSONException("not support type : " + collectionType);
				}
			}

			List<Object> array = new ArrayList<Object>();
			this.parseArray((Class) upperBoundType, array);
			return array;

			// throw new JSONException("not support type : " +
			// collectionType);return parse();
		}

		if (actualTypeArgument instanceof TypeVariable) {
			TypeVariable typeVariable = (TypeVariable) actualTypeArgument;
			Type[] bounds = typeVariable.getBounds();

			if (bounds.length != 1) {
				throw new JSONException("not support : " + typeVariable);
			}

			Type boundType = bounds[0];
			if (boundType instanceof Class) {
				List<Object> array = new ArrayList<Object>();
				this.parseArray((Class) boundType, array);
				return array;
			}
		}

		if (actualTypeArgument instanceof ParameterizedType) {
			ParameterizedType parameterizedType = (ParameterizedType) actualTypeArgument;

			List<Object> array = new ArrayList<Object>();
			this.parseArray(parameterizedType, array);
			return array;
		}

		throw new JSONException("TODO : " + collectionType);
	}
}
