/*
 * Copyright 1999-2101 Alibaba Group.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.meidusa.fastjson.parser;

import java.beans.PropertyDescriptor;
import java.io.File;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Currency;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicIntegerArray;
import java.util.concurrent.atomic.AtomicLongArray;
import java.util.regex.Pattern;

import com.meidusa.fastjson.JSONArray;
import com.meidusa.fastjson.JSONException;
import com.meidusa.fastjson.JSONObject;
import com.meidusa.fastjson.parser.deserializer.ASMDeserializerFactory;
import com.meidusa.fastjson.parser.deserializer.ASMJavaBeanDeserializer;
import com.meidusa.fastjson.parser.deserializer.ArrayDeserializer;
import com.meidusa.fastjson.parser.deserializer.ArrayListStringDeserializer;
import com.meidusa.fastjson.parser.deserializer.ArrayListStringFieldDeserializer;
import com.meidusa.fastjson.parser.deserializer.ArrayListTypeDeserializer;
import com.meidusa.fastjson.parser.deserializer.ArrayListTypeFieldDeserializer;
import com.meidusa.fastjson.parser.deserializer.AtomicIntegerArrayDeserializer;
import com.meidusa.fastjson.parser.deserializer.AtomicLongArrayDeserializer;
import com.meidusa.fastjson.parser.deserializer.AutowiredObjectDeserializer;
import com.meidusa.fastjson.parser.deserializer.BigDecimalDeserializer;
import com.meidusa.fastjson.parser.deserializer.BigIntegerDeserializer;
import com.meidusa.fastjson.parser.deserializer.BooleanDeserializer;
import com.meidusa.fastjson.parser.deserializer.BooleanFieldDeserializer;
import com.meidusa.fastjson.parser.deserializer.ByteDeserializer;
import com.meidusa.fastjson.parser.deserializer.CharacterDeserializer;
import com.meidusa.fastjson.parser.deserializer.CharsetDeserializer;
import com.meidusa.fastjson.parser.deserializer.CollectionDeserializer;
import com.meidusa.fastjson.parser.deserializer.ConcurrentHashMapDeserializer;
import com.meidusa.fastjson.parser.deserializer.CurrencyDeserializer;
import com.meidusa.fastjson.parser.deserializer.DateDeserializer;
import com.meidusa.fastjson.parser.deserializer.DefaultFieldDeserializer;
import com.meidusa.fastjson.parser.deserializer.DefaultObjectDeserializer;
import com.meidusa.fastjson.parser.deserializer.DoubleDeserializer;
import com.meidusa.fastjson.parser.deserializer.EnumDeserializer;
import com.meidusa.fastjson.parser.deserializer.FieldDeserializer;
import com.meidusa.fastjson.parser.deserializer.FileDeserializer;
import com.meidusa.fastjson.parser.deserializer.FloatDeserializer;
import com.meidusa.fastjson.parser.deserializer.HashMapDeserializer;
import com.meidusa.fastjson.parser.deserializer.HashSetStringDeserializer;
import com.meidusa.fastjson.parser.deserializer.HashSetTypeDeserializer;
import com.meidusa.fastjson.parser.deserializer.InetAddressDeserializer;
import com.meidusa.fastjson.parser.deserializer.InetSocketAddressDeserializer;
import com.meidusa.fastjson.parser.deserializer.IntegerDeserializer;
import com.meidusa.fastjson.parser.deserializer.IntegerFieldDeserializer;
import com.meidusa.fastjson.parser.deserializer.JSONArrayDeserializer;
import com.meidusa.fastjson.parser.deserializer.JSONObjectDeserializer;
import com.meidusa.fastjson.parser.deserializer.JavaBeanDeserializer;
import com.meidusa.fastjson.parser.deserializer.JavaObjectDeserializer;
import com.meidusa.fastjson.parser.deserializer.LinkedHashMapDeserializer;
import com.meidusa.fastjson.parser.deserializer.LocaleDeserializer;
import com.meidusa.fastjson.parser.deserializer.LongDeserializer;
import com.meidusa.fastjson.parser.deserializer.LongFieldDeserializer;
import com.meidusa.fastjson.parser.deserializer.NumberDeserializer;
import com.meidusa.fastjson.parser.deserializer.ObjectDeserializer;
import com.meidusa.fastjson.parser.deserializer.PatternDeserializer;
import com.meidusa.fastjson.parser.deserializer.ShortDeserializer;
import com.meidusa.fastjson.parser.deserializer.SqlDateDeserializer;
import com.meidusa.fastjson.parser.deserializer.StringDeserializer;
import com.meidusa.fastjson.parser.deserializer.StringFieldDeserializer;
import com.meidusa.fastjson.parser.deserializer.ThrowableDeserializer;
import com.meidusa.fastjson.parser.deserializer.TimeZoneDeserializer;
import com.meidusa.fastjson.parser.deserializer.TimestampDeserializer;
import com.meidusa.fastjson.parser.deserializer.TreeMapDeserializer;
import com.meidusa.fastjson.parser.deserializer.TypeHandleBeanDeserializer;
import com.meidusa.fastjson.parser.deserializer.URIDeserializer;
import com.meidusa.fastjson.parser.deserializer.URLDeserializer;
import com.meidusa.fastjson.parser.deserializer.UUIDDeserializer;
import com.meidusa.fastjson.util.ASMUtils;
import com.meidusa.fastjson.util.FieldInfo;
import com.meidusa.fastjson.util.IdentityHashMap;
import com.meidusa.fastjson.util.ServiceLoader;
import com.meidusa.fastmark.TypeHandle;
import com.meidusa.fastmark.TypeHandleKey;
import com.meidusa.fastmark.TypeHandleValue;

/**
 * @author wenshao<szujobs@hotmail.com>
 */
public class ParserConfig {

	public static ParserConfig getGlobalInstance() {
		return global;
	}

	private final Set<Class<?>> primitiveClasses = new HashSet<Class<?>>();

	private static ParserConfig global = new ParserConfig();

	private final IdentityHashMap<Type, ObjectDeserializer> derializers = new IdentityHashMap<Type, ObjectDeserializer>();

	private DefaultObjectDeserializer defaultSerializer = new DefaultObjectDeserializer();

	private boolean asmEnable = !ASMUtils.isAndroid();

	protected final SymbolTable symbolTable = new SymbolTable();

	public DefaultObjectDeserializer getDefaultSerializer() {
		return defaultSerializer;
	}

	public ParserConfig() {
		primitiveClasses.add(boolean.class);
		primitiveClasses.add(Boolean.class);

		primitiveClasses.add(char.class);
		primitiveClasses.add(Character.class);

		primitiveClasses.add(byte.class);
		primitiveClasses.add(Byte.class);

		primitiveClasses.add(short.class);
		primitiveClasses.add(Short.class);

		primitiveClasses.add(int.class);
		primitiveClasses.add(Integer.class);

		primitiveClasses.add(long.class);
		primitiveClasses.add(Long.class);

		primitiveClasses.add(float.class);
		primitiveClasses.add(Float.class);

		primitiveClasses.add(double.class);
		primitiveClasses.add(Double.class);

		primitiveClasses.add(BigInteger.class);
		primitiveClasses.add(BigDecimal.class);

		primitiveClasses.add(String.class);
		primitiveClasses.add(java.util.Date.class);
		primitiveClasses.add(java.sql.Date.class);
		primitiveClasses.add(java.sql.Time.class);
		primitiveClasses.add(java.sql.Timestamp.class);

		derializers.put(java.sql.Timestamp.class, TimestampDeserializer.instance);
		derializers.put(java.sql.Date.class, SqlDateDeserializer.instance);
		derializers.put(java.util.Date.class, DateDeserializer.instance);

		derializers.put(JSONObject.class, JSONObjectDeserializer.instance);
		derializers.put(JSONArray.class, JSONArrayDeserializer.instance);

		derializers.put(Map.class, HashMapDeserializer.instance);
		//derializers.put(HashMap.class, HashMapDeserializer.instance);
		//derializers.put(LinkedHashMap.class, LinkedHashMapDeserializer.instance);
		derializers.put(TreeMap.class, TreeMapDeserializer.instance);
		derializers.put(ConcurrentMap.class, ConcurrentHashMapDeserializer.instance);
		//derializers.put(ConcurrentHashMap.class, ConcurrentHashMapDeserializer.instance);

		derializers.put(Collection.class, CollectionDeserializer.instance);
		derializers.put(List.class, CollectionDeserializer.instance);
		derializers.put(ArrayList.class, CollectionDeserializer.instance);

		derializers.put(Object.class, JavaObjectDeserializer.instance);
		derializers.put(String.class, StringDeserializer.instance);
		derializers.put(char.class, CharacterDeserializer.instance);
		derializers.put(Character.class, CharacterDeserializer.instance);
		derializers.put(byte.class, ByteDeserializer.instance);
		derializers.put(Byte.class, ByteDeserializer.instance);
		derializers.put(short.class, ShortDeserializer.instance);
		derializers.put(Short.class, ShortDeserializer.instance);
		derializers.put(int.class, IntegerDeserializer.instance);
		derializers.put(Integer.class, IntegerDeserializer.instance);
		derializers.put(long.class, LongDeserializer.instance);
		derializers.put(Long.class, LongDeserializer.instance);
		derializers.put(BigInteger.class, BigIntegerDeserializer.instance);
		derializers.put(BigDecimal.class, BigDecimalDeserializer.instance);
		derializers.put(float.class, FloatDeserializer.instance);
		derializers.put(Float.class, FloatDeserializer.instance);
		derializers.put(double.class, DoubleDeserializer.instance);
		derializers.put(Double.class, DoubleDeserializer.instance);
		derializers.put(boolean.class, BooleanDeserializer.instance);
		derializers.put(Boolean.class, BooleanDeserializer.instance);
		derializers.put(UUID.class, UUIDDeserializer.instance);
		derializers.put(TimeZone.class, TimeZoneDeserializer.instance);
		derializers.put(Locale.class, LocaleDeserializer.instance);
		derializers.put(InetAddress.class, InetAddressDeserializer.instance);
		derializers.put(Inet4Address.class, InetAddressDeserializer.instance);
		derializers.put(Inet6Address.class, InetAddressDeserializer.instance);
		derializers.put(InetSocketAddress.class, InetSocketAddressDeserializer.instance);
		derializers.put(File.class, FileDeserializer.instance);
		derializers.put(URI.class, URIDeserializer.instance);
		derializers.put(URL.class, URLDeserializer.instance);
		derializers.put(Pattern.class, PatternDeserializer.instance);
		derializers.put(Charset.class, CharsetDeserializer.instance);
		derializers.put(Number.class, NumberDeserializer.instance);
		derializers.put(AtomicIntegerArray.class, AtomicIntegerArrayDeserializer.instance);
		derializers.put(AtomicLongArray.class, AtomicLongArrayDeserializer.instance);
		derializers.put(Currency.class, CurrencyDeserializer.instance);

	}

	public boolean isAsmEnable() {
		return asmEnable;
	}

	public void setAsmEnable(boolean asmEnable) {
		this.asmEnable = asmEnable;
	}

	public SymbolTable getSymbolTable() {
		return symbolTable;
	}

	public IdentityHashMap<Type, ObjectDeserializer> getDerializers() {
		return derializers;
	}

	public ObjectDeserializer getDeserializer(Type type) {
		ObjectDeserializer derializer = this.derializers.get(type);
		if (derializer != null) {
			return derializer;
		}

		if (type instanceof Class<?>) {
			return getDeserializer((Class<?>) type, type);
		}

		if (type instanceof ParameterizedType) {
			Type rawType = ((ParameterizedType) type).getRawType();
			return getDeserializer((Class<?>) rawType, type);
		}

		return this.defaultSerializer;
	}

	public ObjectDeserializer getDeserializer(Class<?> clazz, Type type) {
		ObjectDeserializer derializer = derializers.get(clazz);
		if (derializer != null) {
			return derializer;
		}

		final ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
		for (AutowiredObjectDeserializer autowired : ServiceLoader.load(AutowiredObjectDeserializer.class, classLoader)) {
			for (Type forType : autowired.getAutowiredFor()) {
				derializers.put(forType, autowired);
			}
		}

		derializer = derializers.get(type);
		if (derializer != null) {
			return derializer;
		}

		if (clazz.isEnum()) {
			derializer = new EnumDeserializer(clazz);
		} else if (clazz.isArray()) {
			return ArrayDeserializer.instance;
		} else if (clazz == Set.class) {
			if (type instanceof ParameterizedType) {
				Type itemType = ((ParameterizedType) type).getActualTypeArguments()[0];
				if (itemType == String.class) {
					derializer = HashSetStringDeserializer.instance;
				} else {
					derializer = new HashSetTypeDeserializer(itemType);
				}
			} else {
				derializer = CollectionDeserializer.instance;
			}
		} else if (clazz == Collection.class || clazz == List.class || clazz == ArrayList.class) {

			if (type instanceof ParameterizedType) {
				Type itemType = ((ParameterizedType) type).getActualTypeArguments()[0];
				if (itemType == String.class) {
					derializer = ArrayListStringDeserializer.instance;
				} else {
					derializer = new ArrayListTypeDeserializer(itemType);
				}
			} else {
				derializer = CollectionDeserializer.instance;
			}
		} else if (Collection.class.isAssignableFrom(clazz)) {
			derializer = CollectionDeserializer.instance;
		} else if (Map.class.isAssignableFrom(clazz)) {
			derializer = this.defaultSerializer;
		} else if (Throwable.class.isAssignableFrom(clazz)) {
			derializer = new ThrowableDeserializer(this, clazz);
		} else {
			derializer = createJavaBeanDeserializer(type);
		}

		putDeserializer(type, derializer);

		return derializer;
	}

	public ObjectDeserializer createJavaBeanDeserializer(Type type) {

		Class<?> clazz = null;
		if (type instanceof Class<?>) {
			clazz = (Class<?>) type;
		} else if (type instanceof ParameterizedType) {
			clazz = (Class<?>) ((ParameterizedType) type).getRawType();
		}
		if (clazz == Class.class) {
			return this.defaultSerializer;
		}

		if (!Modifier.isPublic(clazz.getModifiers())) {
			return new JavaBeanDeserializer(this, clazz);
		}

		if (!asmEnable) {
			return new JavaBeanDeserializer(this, clazz);
		}
		TypeHandle typeHandleAnnotation = clazz.getAnnotation(TypeHandle.class);
		if (typeHandleAnnotation != null) {
			String keyField = null;
			keyField = typeHandleAnnotation.key();
			if ("".equals(keyField)) {
				keyField = computeKey(clazz);
			}
			if (keyField != null) {
				Class<?> defaultClass = typeHandleAnnotation.defaultClass();
				if (defaultClass == null || defaultClass == Object.class) {
					defaultClass = clazz;
				}
				TypeHandleBeanDeserializer serializer = new TypeHandleBeanDeserializer(this, defaultClass);
				serializer.setKey(keyField);
				for (int i = 0; i < typeHandleAnnotation.childClasses().length; i++) {
					Class<?> childClass = typeHandleAnnotation.childClasses()[i];
					TypeHandleValue typeHandleValue = childClass.getAnnotation(TypeHandleValue.class);
					if (typeHandleValue != null) {
						serializer.addValueClass(typeHandleValue.value(), childClass);
					} else {
						try {
							Object child = childClass.newInstance();
							PropertyDescriptor descriptor = new PropertyDescriptor(keyField, childClass);
							Method method = descriptor.getReadMethod();
							method.invoke(child);
						} catch (Exception e) {
							// ignore here
						}
					}
				}
				return serializer;
			}
		}
		try {
			return ASMDeserializerFactory.getInstance().createJavaBeanDeserializer(this, type);
		} catch (Exception e) {
			throw new JSONException("create asm deserializer error, " + clazz.getName(), e);
		}
	}

	public static String computeKey(Class<?> clazz) {
		Field[] field = clazz.getDeclaredFields();
		for (int i = 0; i < field.length; i++) {
			TypeHandleKey key = field[i].getAnnotation(TypeHandleKey.class);
			if (key != null) {
				return field[i].getName();
			}
		}
		return null;
	}

	public FieldDeserializer createFieldDeserializer(ParserConfig mapping, Class<?> clazz, FieldInfo fieldInfo) {
		boolean asmEnable = this.asmEnable;

		Method method = fieldInfo.getMethod();

		if (!Modifier.isPublic(clazz.getModifiers())) {
			asmEnable = false;
		}

		if (method.getParameterTypes()[0] == Class.class) {
			asmEnable = false;
		}

		if (!asmEnable) {
			return createFieldDeserializerWithoutASM(mapping, clazz, fieldInfo);
		}

		try {
			return ASMDeserializerFactory.getInstance().createFieldDeserializer(mapping, clazz, fieldInfo);
		} catch (Throwable e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		return createFieldDeserializerWithoutASM(mapping, clazz, fieldInfo);
	}

	public FieldDeserializer createFieldDeserializerWithoutASM(ParserConfig mapping, Class<?> clazz, FieldInfo fieldInfo) {
		Method method = fieldInfo.getMethod();
		Class<?> fieldClass = method.getParameterTypes()[0];

		if (fieldClass == boolean.class || fieldClass == Boolean.class) {
			return new BooleanFieldDeserializer(mapping, clazz, fieldInfo);
		}

		if (fieldClass == int.class || fieldClass == Integer.class) {
			return new IntegerFieldDeserializer(mapping, clazz, fieldInfo);
		}

		if (fieldClass == long.class || fieldClass == Long.class) {
			return new LongFieldDeserializer(mapping, clazz, fieldInfo);
		}

		if (fieldClass == String.class) {
			return new StringFieldDeserializer(mapping, clazz, fieldInfo);
		}

		if (fieldClass == List.class || fieldClass == ArrayList.class) {
			Type fieldType = method.getGenericParameterTypes()[0];
			
			 Type itemType = Object.class;
             if(fieldType instanceof ParameterizedType){
                 itemType = ((ParameterizedType) fieldType).getActualTypeArguments()[0];
             }
             
			if (itemType == String.class) {
				return new ArrayListStringFieldDeserializer(mapping, clazz, fieldInfo);
			}

			return new ArrayListTypeFieldDeserializer(mapping, clazz, fieldInfo);
		}

		return new DefaultFieldDeserializer(mapping, clazz, fieldInfo);
	}

	public void putDeserializer(Type type, ObjectDeserializer deserializer) {
		derializers.put(type, deserializer);
	}

	public ObjectDeserializer getDeserializer(FieldInfo fieldInfo) {
	    Type type = fieldInfo.getFieldType();
	    
		return getDeserializer(fieldInfo.getFieldClass(), type);
	}

	public boolean isPrimitive(Class<?> clazz) {
		return primitiveClasses.contains(clazz);
	}

	public static Field getField(Class<?> clazz, String fieldName) {
		try {
			return clazz.getDeclaredField(fieldName);
		} catch (Exception e) {
			return null;
		}
	}

	public Map<String, FieldDeserializer> getFieldDeserializers(Class<?> clazz) {
		ObjectDeserializer deserizer = getDeserializer(clazz);

		if (deserizer instanceof JavaBeanDeserializer) {
			return ((JavaBeanDeserializer) deserizer).getFieldDeserializerMap();
		} else if (deserizer instanceof ASMJavaBeanDeserializer) {
			return ((ASMJavaBeanDeserializer) deserizer).getInnterSerializer().getFieldDeserializerMap();
		} else {
			return Collections.emptyMap();
		}
	}

}
